<?php
// admin/api/scan_resultados.php
// Escanea carpetas de resultados y actualiza data/resultados.json
// Soporta:
//  A) Estructura nueva: img/resultados/{AÑO}/{TORNEO}/{DISCIPLINA}/1.png
//  B) Estructura legacy: img/{TORNEO}/{DISCIPLINA}/1.png  (recomendado para 2025 ya existente)

header('Content-Type: application/json; charset=utf-8');

$root = realpath(__DIR__ . '/../../');          // /admin
$site = realpath(__DIR__ . '/../../..');       // raíz del sitio (asumiendo /admin/api)
if ($site === false) { http_response_code(500); echo json_encode(['ok'=>false,'error'=>'No se pudo resolver root']); exit; }

$year = isset($_GET['year']) ? trim($_GET['year']) : '';
if ($year === '' || !preg_match('/^\d{4}$/', $year)) { $year = '2025'; }

$imgRoot = $site . '/img';
$dataDir = $site . '/data';
$jsonFile = $dataDir . '/resultados.json';

if (!is_dir($dataDir)) @mkdir($dataDir, 0775, true);

function load_json($path){
  if (!file_exists($path)) return ['title'=>'Resultados', 'years'=>[]];
  $raw = @file_get_contents($path);
  $j = json_decode($raw, true);
  if (!is_array($j)) return ['title'=>'Resultados', 'years'=>[]];
  if (!isset($j['title'])) $j['title'] = 'Resultados';
  if (!isset($j['years']) || !is_array($j['years'])) $j['years'] = [];
  return $j;
}

function save_json($path, $data){
  $tmp = $path . '.tmp';
  $ok = @file_put_contents($tmp, json_encode($data, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE));
  if ($ok === false) return false;
  return @rename($tmp, $path);
}

function is_tournament_dir($dir){
  // Un torneo válido tiene al menos una subcarpeta disciplina con 1.png
  $disciplines = ['benchpress','deadlift','pushpull'];
  foreach ($disciplines as $d){
    if (file_exists($dir . '/' . $d . '/1.png')) return true;
  }
  return false;
}

function scan_structure_new($baseDir){
  // baseDir: img/resultados/{year}
  $events = [];
  if (!is_dir($baseDir)) return $events;

  $tournamentDirs = glob($baseDir . '/*', GLOB_ONLYDIR);
  if (!$tournamentDirs) return $events;

  foreach ($tournamentDirs as $td){
    $slug = basename($td);
    $disciplines = [];
    foreach (['benchpress','deadlift','pushpull'] as $d){
      if (file_exists($td . '/' . $d . '/1.png')) $disciplines[] = $d;
    }
    if (!$disciplines) continue;
    $events[$slug] = [
      'slug' => $slug,
      'label' => ucwords(str_replace(['-','_'], ' ', $slug)),
      'date' => '',
      'disciplines' => $disciplines
    ];
  }
  return $events;
}

function scan_structure_legacy($imgRoot){
  $events = [];
  if (!is_dir($imgRoot)) return $events;

  $dirs = glob($imgRoot . '/*', GLOB_ONLYDIR);
  if (!$dirs) return $events;

  foreach ($dirs as $d){
    $slug = basename($d);

    // excluir carpetas típicas que no son torneos
    if (in_array($slug, ['records','record','recordes','assets','css','js','icons'], true)) continue;

    // debe tener estructura de torneo
    if (!is_tournament_dir($d)) continue;

    $disciplines = [];
    foreach (['benchpress','deadlift','pushpull'] as $disc){
      if (file_exists($d . '/' . $disc . '/1.png')) $disciplines[] = $disc;
    }
    if (!$disciplines) continue;

    $events[$slug] = [
      'slug' => $slug,
      'label' => ucwords(str_replace(['-','_'], ' ', $slug)),
      'date' => '',
      'disciplines' => $disciplines
    ];
  }
  return $events;
}

$j = load_json($jsonFile);
if (!isset($j['years'][$year]) || !is_array($j['years'][$year])) {
  $j['years'][$year] = ['title' => 'Resultados ' . $year, 'events' => []];
}
if (!isset($j['years'][$year]['events']) || !is_array($j['years'][$year]['events'])) $j['years'][$year]['events'] = [];

$found = [];

// 1) Preferir estructura nueva si existe para ese año
$newBase = $imgRoot . '/resultados/' . $year;
$found = scan_structure_new($newBase);

// 2) Si no encontró nada y/o querés migrar 2025 legacy, escanear legacy
if (count($found) === 0 || $year === '2025') {
  $legacy = scan_structure_legacy($imgRoot);
  // merge: no pisar los que ya venían de la estructura nueva
  foreach ($legacy as $k => $v){
    if (!isset($found[$k])) $found[$k] = $v;
  }
}

// Merge con lo existente para mantener labels/fechas editadas en JSON
$existingBySlug = [];
foreach ($j['years'][$year]['events'] as $ev){
  if (isset($ev['slug'])) $existingBySlug[$ev['slug']] = $ev;
}

$outEvents = [];
foreach ($found as $slug => $ev){
  if (isset($existingBySlug[$slug])) {
    // conservar label/date si estaban cargados
    $prev = $existingBySlug[$slug];
    if (!empty($prev['label'])) $ev['label'] = $prev['label'];
    if (!empty($prev['date']))  $ev['date']  = $prev['date'];
    // disciplinas: merge
    if (isset($prev['disciplines']) && is_array($prev['disciplines'])) {
      $ev['disciplines'] = array_values(array_unique(array_merge($prev['disciplines'], $ev['disciplines'])));
    }
  }
  $outEvents[] = $ev;
}

// ordenar por fecha (si hay), si no por label
usort($outEvents, function($a,$b){
  $da = $a['date'] ?? '';
  $db = $b['date'] ?? '';
  if ($da === $db) return strcmp($a['label'] ?? '', $b['label'] ?? '');
  if ($da === '') return 1;
  if ($db === '') return -1;
  return strcmp($da, $db);
});

$j['years'][$year]['events'] = $outEvents;

$ok = save_json($jsonFile, $j);
if (!$ok){
  http_response_code(500);
  echo json_encode(['ok'=>false,'error'=>'No se pudo escribir resultados.json (permisos?)','year'=>$year], JSON_UNESCAPED_UNICODE);
  exit;
}

echo json_encode(['ok'=>true,'year'=>$year,'events'=>count($outEvents),'mode'=> (is_dir($newBase)?'new+legacy':'legacy')], JSON_UNESCAPED_UNICODE);
