<?php
// admin/panel.php - Panel simple (sin DB) para cargar PDFs/PNGs y mantener JSON
// Requiere PHP con permisos de escritura en /data y /img.

// ====== CONFIG ======
$BASE_DIR   = realpath(__DIR__ . '/..'); // raíz del sitio
$DATA_DIR   = $BASE_DIR . '/data';
$IMG_DIR    = $BASE_DIR . '/img';
$RECORDS_JSON = $DATA_DIR . '/records.json';
$RESULTS_JSON = $DATA_DIR . '/resultados.json';

// Opcional: protección simple (si querés)
// Si ya tenés login en tu hosting, podés borrar esto y dejar tu sistema.
// $PASS = 'cambiame';
// if (!isset($_SERVER['PHP_AUTH_USER'])) {
//   header('WWW-Authenticate: Basic realm="APL Admin"');
//   header('HTTP/1.0 401 Unauthorized');
//   exit('Auth requerida');
// }

function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function read_json_file($path, $fallback){
  if (!file_exists($path)) return $fallback;
  $raw = file_get_contents($path);
  $j = json_decode($raw, true);
  return is_array($j) ? $j : $fallback;
}
function write_json_file($path, $data){
  $dir = dirname($path);
  if (!is_dir($dir)) mkdir($dir, 0775, true);
  $tmp = $path . '.tmp';
  file_put_contents($tmp, json_encode($data, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
  rename($tmp, $path);
}
function ensure_dir($path){
  if (!is_dir($path)) mkdir($path, 0775, true);
}
function safe_slug($s){
  $s = strtolower(trim((string)$s));
  $s = preg_replace('/[^a-z0-9\-_]+/','', str_replace(' ','', $s));
  return $s ?: 'torneo';
}
function move_upload($file, $dest){
  if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) return [false, 'No se subió archivo.'];
  ensure_dir(dirname($dest));
  if (!move_uploaded_file($file['tmp_name'], $dest)) return [false, 'No se pudo mover el archivo.'];
  return [true, 'OK'];
}

$recordsFallback = [
  "updated_at" => date('Y-m-d'),
  "title" => "Récords APUA WABDL",
  "subtitle" => "Elegí disciplina y categoría. Se abre el <strong>PDF</strong>.",
  "groups" => []
];

$resultadosFallback = [
  "title" => "Resultados",
  "subtitle" => "Elegí año, torneo y disciplina. Se abre el <strong>PNG</strong>.",
  "years" => []
];

$records = read_json_file($RECORDS_JSON, $recordsFallback);
$resultados = read_json_file($RESULTS_JSON, $resultadosFallback);

$msg = null; $err = null;

// ====== ACCIONES ======
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $action = $_POST['action'] ?? '';

  // --- Crear año (Resultados) ---
  if ($action === 'create_year') {
    $year = trim($_POST['year'] ?? '');
    if (!preg_match('/^\d{4}$/', $year)) {
      $err = 'Año inválido.';
    } else {
      if (!isset($resultados['years'][$year])) {
        $resultados['years'][$year] = [
          "title" => "Resultados $year",
          "events" => []
        ];
        write_json_file($RESULTS_JSON, $resultados);
        $msg = "Año $year creado.";
      } else {
        $msg = "El año $year ya existe (lo podés usar).";
      }
    }
  }

  // --- Subir/Actualizar PDF (Records) ---
  if ($action === 'upload_record_pdf') {
    $disc = trim($_POST['disciplina'] ?? '');
    $cat  = trim($_POST['categoria'] ?? '');
    if ($disc === '' || $cat === '') {
      $err = 'Falta disciplina o categoría.';
    } elseif (!isset($_FILES['pdf']) || $_FILES['pdf']['error'] !== UPLOAD_ERR_OK) {
      $err = 'Falta PDF o hubo error de subida.';
    } else {
      $dest = $IMG_DIR . "/records/$disc/$cat/1.pdf";
      [$ok, $why] = move_upload($_FILES['pdf'], $dest);
      if (!$ok) $err = $why;
      else {
        $records['updated_at'] = date('Y-m-d');

        // asegurar que exista en JSON (si no, lo agrega)
        $foundG = false;
        foreach ($records['groups'] as &$g) {
          if (($g['key'] ?? '') === $disc) {
            $foundG = true;
            $items = $g['items'] ?? [];
            $exists = false;
            foreach ($items as $it) if (($it['key'] ?? '') === $cat) $exists = true;
            if (!$exists) {
              $items[] = ["key"=>$cat, "label"=>ucwords(str_replace(['_','-'],' ', $cat))];
              $g['items'] = $items;
            }
          }
        }
        unset($g);

        if (!$foundG) {
          $records['groups'][] = [
            "key" => $disc,
            "label" => ucwords(str_replace(['_','-'],' ', $disc)),
            "subtitle" => "APUA WABDL",
            "items" => [[ "key"=>$cat, "label"=>ucwords(str_replace(['_','-'],' ', $cat)) ]]
          ];
        }

        write_json_file($RECORDS_JSON, $records);
        $msg = "PDF subido: img/records/$disc/$cat/1.pdf";
      }
    }
  }

  // --- Subir/Actualizar PNG (Resultados) ---
  if ($action === 'upload_result_png') {
    $year = trim($_POST['year'] ?? '');
    $slug = safe_slug($_POST['slug'] ?? '');
    $label = trim($_POST['label'] ?? '');
    $date  = trim($_POST['date'] ?? '');
    $disc  = trim($_POST['disciplina'] ?? '');

    if (!preg_match('/^\d{4}$/', $year)) $err = 'Año inválido.';
    elseif ($slug === '' || $disc === '') $err = 'Falta torneo (carpeta) o disciplina.';
    elseif (!isset($_FILES['png']) || $_FILES['png']['error'] !== UPLOAD_ERR_OK) $err = 'Falta PNG o hubo error de subida.';
    else {
      // crear año si no existe
      if (!isset($resultados['years'][$year])) {
        $resultados['years'][$year] = ["title"=>"Resultados $year", "events"=>[]];
      }
      $dest = $IMG_DIR . "/$slug/$disc/1.png";
      [$ok, $why] = move_upload($_FILES['png'], $dest);
      if (!$ok) $err = $why;
      else {
        // asegurar evento en JSON
        $events = $resultados['years'][$year]['events'] ?? [];
        $idx = null;
        foreach ($events as $i=>$ev) {
          if (($ev['slug'] ?? '') === $slug) { $idx = $i; break; }
        }
        if ($idx === null) {
          $events[] = [
            "slug"=>$slug,
            "label"=>$label ?: ucfirst($slug),
            "date"=>$date,
            "disciplines"=>[]
          ];
          $idx = count($events)-1;
        } else {
          if ($label) $events[$idx]['label'] = $label;
          if ($date)  $events[$idx]['date']  = $date;
        }

        $discList = $events[$idx]['disciplines'] ?? [];
        $has = false;
        foreach ($discList as $d) {
          if (is_array($d) && ($d['key'] ?? '') === $disc) $has = true;
          if (is_string($d) && $d === $disc) $has = true;
        }
        if (!$has) $discList[] = ["key"=>$disc, "label"=>ucwords(str_replace(['_','-'],' ', $disc))];
        $events[$idx]['disciplines'] = $discList;

        $resultados['years'][$year]['events'] = $events;
        write_json_file($RESULTS_JSON, $resultados);
        $msg = "PNG subido: img/$slug/$disc/1.png (año $year)";
      }
    }
  }

  // recargar en memoria
  $records = read_json_file($RECORDS_JSON, $recordsFallback);
  $resultados = read_json_file($RESULTS_JSON, $resultadosFallback);
}

// Years list
$years = array_keys($resultados['years'] ?? []);
rsort($years);
$defaultYear = $years[0] ?? date('Y');

?><!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Panel APL (sin DB)</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body{background:#f6f9ff;}
    .cardx{background:#fff;border:1px solid rgba(0,0,0,.08);border-radius:18px;box-shadow:0 16px 34px rgba(0,0,0,.06)}
    code{background:rgba(0,0,0,.06);padding:.15rem .35rem;border-radius:.4rem}
    .smallhint{color:#5b6b7b}
  </style>
</head>
<body>
<nav class="navbar navbar-expand-lg bg-white border-bottom">
  <div class="container">
    <a class="navbar-brand fw-bold" href="#">Panel APL</a>
    <div class="ms-auto d-flex gap-2">
      <a class="btn btn-outline-primary" href="../index.html" target="_blank">Ver sitio</a>
    </div>
  </div>
</nav>

<main class="container py-4">
  <?php if ($msg): ?>
    <div class="alert alert-success"><?=h($msg)?></div>
  <?php endif; ?>
  <?php if ($err): ?>
    <div class="alert alert-danger"><?=h($err)?></div>
  <?php endif; ?>

  <div class="row g-4">
    <!-- RECORDS -->
    <div class="col-lg-6">
      <div class="cardx p-4 h-100">
        <h4 class="fw-bold mb-1">Récords (PDF)</h4>
        <div class="smallhint mb-3">
          Estructura: <code>img/records/{disciplina}/{categoria}/1.pdf</code> · Datos: <code>data/records.json</code>
        </div>

        <form method="post" enctype="multipart/form-data" class="row g-3">
          <input type="hidden" name="action" value="upload_record_pdf">

          <div class="col-md-6">
            <label class="form-label fw-bold">Disciplina</label>
            <select name="disciplina" class="form-select" required>
              <option value="benchpress">benchpress</option>
              <option value="deadlift">deadlift</option>
              <option value="pushpull">pushpull</option>
            </select>
          </div>

          <div class="col-md-6">
            <label class="form-label fw-bold">Categoría (carpeta)</label>
            <input name="categoria" class="form-control" placeholder="ej: benchpressraw" required>
            <div class="form-text">Debe coincidir con la carpeta dentro de la disciplina.</div>
          </div>

          <div class="col-12">
            <label class="form-label fw-bold">Subir PDF (se guarda como 1.pdf)</label>
            <input type="file" name="pdf" accept="application/pdf" class="form-control" required>
          </div>

          <div class="col-12">
            <button class="btn btn-primary fw-bold">Subir / Reemplazar</button>
          </div>
        </form>

        <hr class="my-4">
        <div class="fw-bold mb-2">Estado (records.json)</div>
        <textarea class="form-control" rows="10" readonly><?php
          echo json_encode($records, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE);
        ?></textarea>
      </div>
    </div>

    <!-- RESULTADOS -->
    <div class="col-lg-6">
      <div class="cardx p-4 h-100">
        <h4 class="fw-bold mb-1">Resultados (PNG) · por año</h4>
        <div class="smallhint mb-3">
          Estructura: <code>img/{torneo}/{disciplina}/1.png</code> · Datos: <code>data/resultados.json</code>
        </div>

        <form method="post" class="row g-3 mb-3">
          <input type="hidden" name="action" value="create_year">
          <div class="col-md-6">
            <label class="form-label fw-bold">Crear año</label>
            <input name="year" class="form-control" placeholder="2026" inputmode="numeric" pattern="\d{4}" required>
          </div>
          <div class="col-md-6 d-flex align-items-end">
            <button class="btn btn-outline-primary fw-bold w-100">Crear (si no existe)</button>
          </div>
        </form>

        <form method="post" enctype="multipart/form-data" class="row g-3">
          <input type="hidden" name="action" value="upload_result_png">

          <div class="col-md-6">
            <label class="form-label fw-bold">Año</label>
            <select name="year" class="form-select">
              <?php foreach($years as $y): ?>
                <option value="<?=h($y)?>" <?=($y===$defaultYear?'selected':'')?>><?=h($y)?></option>
              <?php endforeach; ?>
              <?php if (!$years): ?>
                <option value="<?=h($defaultYear)?>"><?=h($defaultYear)?></option>
              <?php endif; ?>
            </select>
            <div class="form-text">Si no existe, se crea automáticamente.</div>
          </div>

          <div class="col-md-6">
            <label class="form-label fw-bold">Disciplina</label>
            <select name="disciplina" class="form-select" required>
              <option value="benchpress">benchpress</option>
              <option value="deadlift">deadlift</option>
              <option value="pushpull">pushpull</option>
            </select>
          </div>

          <div class="col-md-6">
            <label class="form-label fw-bold">Carpeta torneo (slug)</label>
            <input name="slug" class="form-control" placeholder="miramar" required>
            <div class="form-text">Se usará como carpeta en <code>img/</code>.</div>
          </div>

          <div class="col-md-6">
            <label class="form-label fw-bold">Nombre visible</label>
            <input name="label" class="form-control" placeholder="Miramar">
          </div>

          <div class="col-12">
            <label class="form-label fw-bold">Fecha (texto)</label>
            <input name="date" class="form-control" placeholder="19/4/2025">
          </div>

          <div class="col-12">
            <label class="form-label fw-bold">Subir PNG (se guarda como 1.png)</label>
            <input type="file" name="png" accept="image/png" class="form-control" required>
          </div>

          <div class="col-12">
            <button class="btn btn-primary fw-bold">Subir / Reemplazar</button>
          </div>
        </form>

        <hr class="my-4">
        <div class="fw-bold mb-2">Estado (resultados.json)</div>
        <textarea class="form-control" rows="10" readonly><?php
          echo json_encode($resultados, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE);
        ?></textarea>
      </div>
    </div>
  </div>

  <div class="mt-4 small text-muted">
    <strong>Nota:</strong> El sitio público (<code>index.html</code>) lee estos JSON por <code>fetch()</code>. Si ves que no carga en producción,
    revisá que el servidor permita servir <code>.json</code> (MIME) y que exista la carpeta <code>data/</code>.
  </div>
</main>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
