<?php
@session_start();

// DEBUG: mostrar errores mientras desarrollamos
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Este archivo está en /admin, subimos un nivel para incluir db.php
require_once __DIR__ . '/../db.php';
$pdo = pdo();

function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }

/* =========================
   1) ID de simulación
   ========================= */
$sim_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($sim_id <= 0) {
    echo "Falta ID de simulación.";
    exit;
}

$error_estado = '';

/* =========================
   2) Cambiar estado (admin)
   ========================= */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['accion_estado'])) {
    $accion = $_POST['accion_estado'];

    // Estados válidos según ENUM
    $estados_validos = ['pendiente','vista','contactado','convertida','descartada'];

    if (in_array($accion, $estados_validos, true)) {

        try {
            // Traigo la simulación completa porque la voy a usar para crear la operación si hace falta
            $stmtSim = $pdo->prepare("SELECT * FROM simulaciones_operacion WHERE id = :id LIMIT 1");
            $stmtSim->execute([':id' => $sim_id]);
            $simRow = $stmtSim->fetch();

            if (!$simRow) {
                throw new Exception("Simulación no encontrada al intentar actualizar estado.");
            }

            // Si la acción es convertir en operación
            if ($accion === 'convertida') {
                $pdo->beginTransaction();

                // Si todavía no tiene operacion_id, la creamos
                if (empty($simRow['operacion_id'])) {

                    // Calculamos saldo a financiar aproximado
                    $precio_lista    = (float)$simRow['precio_referencia'];
                    $senia           = (float)$simRow['senia_deseada'];
                    $ent_veh         = (float)$simRow['entrega_vehiculo_deseada'];
                    $ent_terr        = (float)$simRow['entrega_terreno_deseada'];
                    $ent_cheq        = (float)$simRow['entrega_cheques_deseada'];
                    $cuotas_cant     = (int)$simRow['cuotas_deseadas'];
                    $cuota_monto_est = (float)$simRow['cuota_monto_estimado'];

                    $saldo = $precio_lista - $senia - $ent_veh - $ent_terr - $ent_cheq;
                    if ($saldo < 0) $saldo = 0;

                    // Insert en operaciones (incluyendo codigo_operacion provisorio)
                    $insOp = $pdo->prepare("
                        INSERT INTO operaciones
                            (cliente_id,
                             codigo_operacion,
                             modelo,
                             metros_cubiertos,
                             metros_galeria,
                             sistema,
                             forma_venta,
                             precio_lista,
                             senia,
                             entrega_vehiculo,
                             entrega_terreno,
                             entrega_cheques,
                             saldo_financiar,
                             cuotas_cantidad,
                             cuota_monto,
                             tipo_plan,
                             porcentaje_inicio_obra,
                             estado_obra,
                             observaciones)
                        VALUES
                            (:cliente_id,
                             :codigo_operacion,
                             :modelo,
                             :m_cubiertos,
                             :m_galeria,
                             :sistema,
                             :forma_venta,
                             :precio_lista,
                             :senia,
                             :ent_veh,
                             :ent_terr,
                             :ent_cheq,
                             :saldo,
                             :cuotas,
                             :cuota_monto,
                             'ninguno',
                             60,
                             'pendiente',
                             :observaciones)
                    ");

                    $insOp->execute([
                        ':cliente_id'        => $simRow['cliente_id'],
                        ':codigo_operacion'  => 'PENDIENTE',
                        ':modelo'            => $simRow['modelo'],
                        ':m_cubiertos'       => $simRow['metros_cubiertos'],
                        ':m_galeria'         => $simRow['metros_galeria'],
                        ':sistema'           => $simRow['sistema'],
                        ':forma_venta'       => $simRow['forma_venta_deseada'],
                        ':precio_lista'      => $precio_lista,
                        ':senia'             => $senia,
                        ':ent_veh'           => $ent_veh,
                        ':ent_terr'          => $ent_terr,
                        ':ent_cheq'          => $ent_cheq,
                        ':saldo'             => $saldo,
                        ':cuotas'            => $cuotas_cant,
                        ':cuota_monto'       => $cuota_monto_est,
                        ':observaciones'     => $simRow['comentarios_cliente']
                    ]);

                    $operacion_id = (int)$pdo->lastInsertId();

                    // Genero código de operación tipo OP-000001
                    $codigo_operacion = 'OP-' . str_pad($operacion_id, 6, '0', STR_PAD_LEFT);

                    $updOpCode = $pdo->prepare("
                        UPDATE operaciones
                        SET codigo_operacion = :codigo
                        WHERE id = :id
                    ");
                    $updOpCode->execute([
                        ':codigo' => $codigo_operacion,
                        ':id'     => $operacion_id
                    ]);

                    // Vinculo la simulación con la operación recién creada
                    $updSimOp = $pdo->prepare("
                        UPDATE simulaciones_operacion
                        SET operacion_id = :op_id
                        WHERE id = :id
                    ");
                    $updSimOp->execute([
                        ':op_id' => $operacion_id,
                        ':id'    => $sim_id
                    ]);
                }

                // En cualquier caso, dejo el estado como convertida
                $updEstado = $pdo->prepare("
                    UPDATE simulaciones_operacion
                    SET estado = 'convertida',
                        updated_at = NOW()
                    WHERE id = :id
                ");
                $updEstado->execute([':id' => $sim_id]);

                $pdo->commit();

            } else {
                // Cualquier otro estado: solo actualizo estado
                $upd = $pdo->prepare("
                    UPDATE simulaciones_operacion
                    SET estado = :estado,
                        updated_at = NOW()
                    WHERE id = :id
                ");
                $upd->execute([
                    ':estado' => $accion,
                    ':id'     => $sim_id
                ]);
            }

        } catch (Exception $e) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            $error_estado = "Error al actualizar estado / operación: " . h($e->getMessage());
        }
    }
}

/* =========================
   3) Traer simulación + cliente (actualizada)
   ========================= */
try {
    $sql = "
        SELECT s.*,
               c.nombre, c.apellido, c.dni, c.email, c.telefono, c.localidad
        FROM simulaciones_operacion s
        JOIN clientes c ON c.id = s.cliente_id
        WHERE s.id = :id
        LIMIT 1
    ";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([':id' => $sim_id]);
    $sim = $stmt->fetch();

    if (!$sim) {
        echo "Simulación no encontrada.";
        exit;
    }
} catch (Exception $e) {
    echo "Error al leer la simulación: " . h($e->getMessage());
    exit;
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
<meta charset="UTF-8">
<title>Simulación #<?php echo $sim['id']; ?> - Cabañas Santiago</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<style>
body{background:#f7f7f7;font-family:"Inter",system-ui,sans-serif;}
.badge-estado{font-size:.8rem;}
</style>
</head>
<body>

<div class="container py-4">
    <a href="simulaciones.php" class="btn btn-link mb-3">&larr; Volver a simulaciones</a>

    <?php if (!empty($error_estado)): ?>
        <div class="alert alert-danger"><?php echo $error_estado; ?></div>
    <?php endif; ?>

    <div class="card mb-3">
        <div class="card-header d-flex justify-content-between align-items-center">
            <div>
                <strong>Simulación #<?php echo $sim['id']; ?></strong>
                <span class="text-muted ms-2"><?php echo h($sim['created_at']); ?></span>
                <?php if (!empty($sim['operacion_id'])): ?>
                    <br><small>Operación vinculada: ID <?php echo (int)$sim['operacion_id']; ?></small>
                <?php endif; ?>
            </div>
            <div class="d-flex align-items-center gap-2">
                <span class="badge bg-secondary badge-estado">
                    Estado: <?php echo h($sim['estado']); ?>
                </span>

                <!-- Botones de cambio de estado -->
                <form method="post" class="d-flex gap-1 mb-0">
                    <button type="submit" name="accion_estado" value="vista"
                            class="btn btn-outline-secondary btn-sm"
                            <?php echo ($sim['estado']=='vista'?'disabled':''); ?>>
                        Vista
                    </button>
                    <button type="submit" name="accion_estado" value="contactado"
                            class="btn btn-outline-primary btn-sm"
                            <?php echo ($sim['estado']=='contactado'?'disabled':''); ?>>
                        Contactado
                    </button>
                    <button type="submit" name="accion_estado" value="convertida"
                            class="btn btn-success btn-sm"
                            <?php echo ($sim['estado']=='convertida'?'disabled':''); ?>>
                        Marcar como operación
                    </button>
                    <button type="submit" name="accion_estado" value="descartada"
                            class="btn btn-outline-danger btn-sm"
                            <?php echo ($sim['estado']=='descartada'?'disabled':''); ?>>
                        Descartar
                    </button>
                </form>
            </div>
        </div>

        <div class="card-body">
            <!-- Datos del modelo -->
            <h5 class="mb-3">Modelo solicitado</h5>
            <p class="mb-1"><strong>Modelo:</strong> <?php echo h($sim['modelo']); ?></p>
            <p class="mb-1">
                <strong>Metros:</strong>
                <?php echo h($sim['metros_cubiertos']); ?> m² cubiertos
                <?php if ($sim['metros_galeria'] > 0): ?>
                    + <?php echo h($sim['metros_galeria']); ?> m² galería
                <?php endif; ?>
            </p>
            <p class="mb-1"><strong>Sistema:</strong> <?php echo h($sim['sistema']); ?></p>
            <p class="mb-3"><strong>Precio de referencia:</strong>
                $<?php echo number_format($sim['precio_referencia'],0,',','.'); ?>
            </p>

            <hr>

            <!-- Datos del cliente -->
            <h5 class="mb-3">Datos del cliente</h5>
            <p class="mb-1"><strong>Nombre:</strong> <?php echo h($sim['nombre'].' '.$sim['apellido']); ?></p>
            <p class="mb-1"><strong>DNI:</strong> <?php echo h($sim['dni']); ?></p>
            <p class="mb-1"><strong>Teléfono:</strong> <?php echo h($sim['telefono']); ?></p>
            <p class="mb-1"><strong>Email:</strong> <?php echo h($sim['email']); ?></p>
            <p class="mb-3"><strong>Localidad:</strong> <?php echo h($sim['localidad']); ?></p>

            <hr>

            <!-- Preferencia de pago -->
            <h5 class="mb-3">Cómo quiere pagar</h5>
            <p class="mb-1"><strong>Forma de pago deseada:</strong>
                <?php echo h($sim['forma_venta_deseada']); ?>
            </p>
            <p class="mb-1"><strong>Seña que podría dar:</strong>
                $<?php echo number_format($sim['senia_deseada'],0,',','.'); ?>
            </p>
            <p class="mb-3"><strong>Cantidad de cuotas deseadas:</strong>
                <?php echo (int)$sim['cuotas_deseadas']; ?>
            </p>

            <hr>

            <!-- Entregas como parte de pago -->
            <h5 class="mb-3">Entregas como parte de pago</h5>
            <p class="mb-1">
                <strong>Vehículo:</strong><br>
                <?php echo nl2br(h($sim['entrega_vehiculo_detalle'] ?: '—')); ?>
            </p>
            <p class="mb-1 mt-2">
                <strong>Terreno:</strong><br>
                <?php echo nl2br(h($sim['entrega_terreno_detalle'] ?: '—')); ?>
            </p>
            <p class="mb-1 mt-2">
                <strong>Cheques (monto deseado):</strong>
                $<?php echo number_format($sim['entrega_cheques_deseada'],0,',','.'); ?>
            </p>

            <?php if (!empty($sim['comentarios_cliente'])): ?>
                <hr>
                <h5 class="mb-3">Comentarios del cliente</h5>
                <p><?php echo nl2br(h($sim['comentarios_cliente'])); ?></p>
            <?php endif; ?>
        </div>
    </div>
</div>

</body>
</html>
