<?php
// cliente/crear_operacion.php
@session_start();
if (empty($_SESSION['cliente_id'])) {
    header('Location: login.php');
    exit;
}

require_once '../db.php';
$pdo = pdo();

$cliente_id = (int)$_SESSION['cliente_id'];

// Traer datos del cliente
$stmt = $pdo->prepare("SELECT * FROM clientes WHERE id = :id LIMIT 1");
$stmt->execute([':id' => $cliente_id]);
$cli = $stmt->fetch();
if (!$cli) {
    session_unset();
    session_destroy();
    header('Location: login.php');
    exit;
}

$errores = [];

// Valores por defecto
$datos = [
    'modelo'                 => '',
    'metros_cubiertos'       => '',
    'metros_galeria'         => '',
    'sistema'                => 'estructura',
    'forma_venta'            => 'contado',
    'precio_lista'           => '',
    'senia'                  => '0',
    'entrega_vehiculo'       => '0',
    'entrega_terreno'        => '0',
    'entrega_cheques'        => '0',
    'saldo_financiar'        => '',
    'cuotas_cantidad'        => '',
    'cuota_monto'            => '',
    'tipo_plan'              => 'ninguno',
    'porcentaje_inicio_obra' => '60',
];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Tomar datos
    foreach ($datos as $k => $v) {
        if (isset($_POST[$k])) {
            $datos[$k] = trim($_POST[$k]);
        }
    }

    // Validaciones mínimas
    if ($datos['modelo'] === '') {
        $errores[] = 'Debés indicar qué modelo de casa compraste.';
    }

    $precio_lista = (float)($datos['precio_lista'] ?: 0);
    if ($precio_lista <= 0) {
        $errores[] = 'El precio total de la vivienda debe ser mayor a 0.';
    }

    $metros_cubiertos = (float)($datos['metros_cubiertos'] ?: 0);
    if ($metros_cubiertos <= 0) {
        $errores[] = 'Indicá los metros cubiertos de la casa.';
    }

    // Números
    $senia            = (float)($datos['senia'] ?: 0);
    $ent_vehiculo     = (float)($datos['entrega_vehiculo'] ?: 0);
    $ent_terreno      = (float)($datos['entrega_terreno'] ?: 0);
    $ent_cheques      = (float)($datos['entrega_cheques'] ?: 0);
    $entregas_iniciales = $senia + $ent_vehiculo + $ent_terreno + $ent_cheques;

    $saldo_financiar = $datos['saldo_financiar'] !== ''
        ? (float)$datos['saldo_financiar']
        : max($precio_lista - $entregas_iniciales, 0);

    $cuotas_cantidad = (int)($datos['cuotas_cantidad'] ?: 0);
    $cuota_monto     = (float)($datos['cuota_monto'] ?: 0);

    // Si hay saldo a financiar y cantidad de cuotas pero no monto, lo calculamos
    if ($saldo_financiar > 0 && $cuotas_cantidad > 0 && $cuota_monto <= 0) {
        $cuota_monto = $saldo_financiar / $cuotas_cantidad;
    }

    if ($saldo_financiar < 0) {
        $errores[] = 'El saldo a financiar no puede ser negativo. Revisá los montos.';
    }

    if ($saldo_financiar > 0 && $cuotas_cantidad <= 0) {
        $errores[] = 'Si hay saldo a financiar, indicá la cantidad de cuotas.';
    }

    if (empty($errores)) {
        try {
            // Generar código de operación tipo CS-OP-AAAAMMDDHHMMSS
            $codigo_operacion = 'CS-OP-' . date('YmdHis');

            $stmt = $pdo->prepare("
                INSERT INTO operaciones
                (
                    cliente_id, codigo_operacion, modelo,
                    metros_cubiertos, metros_galeria,
                    sistema, forma_venta,
                    precio_lista, senia,
                    entrega_vehiculo, entrega_terreno, entrega_cheques,
                    saldo_financiar, cuotas_cantidad, cuota_monto,
                    tipo_plan, porcentaje_inicio_obra,
                    estado_obra, fecha_estimada_entrega, fecha_entrega_real,
                    observaciones, fecha_alta, created_at
                )
                VALUES
                (
                    :cliente_id, :codigo_operacion, :modelo,
                    :metros_cubiertos, :metros_galeria,
                    :sistema, :forma_venta,
                    :precio_lista, :senia,
                    :entrega_vehiculo, :entrega_terreno, :entrega_cheques,
                    :saldo_financiar, :cuotas_cantidad, :cuota_monto,
                    :tipo_plan, :porcentaje_inicio_obra,
                    'pendiente', NULL, NULL,
                    :observaciones, NOW(), NOW()
                )
            ");

            $stmt->execute([
                ':cliente_id'            => $cliente_id,
                ':codigo_operacion'      => $codigo_operacion,
                ':modelo'                => $datos['modelo'],
                ':metros_cubiertos'      => $metros_cubiertos,
                ':metros_galeria'        => (float)($datos['metros_galeria'] ?: 0),
                ':sistema'               => $datos['sistema'],
                ':forma_venta'           => $datos['forma_venta'],
                ':precio_lista'          => $precio_lista,
                ':senia'                 => $senia,
                ':entrega_vehiculo'      => $ent_vehiculo,
                ':entrega_terreno'       => $ent_terreno,
                ':entrega_cheques'       => $ent_cheques,
                ':saldo_financiar'       => $saldo_financiar,
                ':cuotas_cantidad'       => $cuotas_cantidad,
                ':cuota_monto'           => $cuota_monto,
                ':tipo_plan'             => $datos['tipo_plan'],
                ':porcentaje_inicio_obra'=> (float)$datos['porcentaje_inicio_obra'],
                ':observaciones'         => 'Operación creada por el cliente desde el portal.',
            ]);

            // Volvemos al panel con aviso
            header('Location: panel.php?op_creada=1');
            exit;

        } catch (PDOException $e) {
            $errores[] = 'Error al crear la operación: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Registrar mi compra - Cabañas Santiago</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">

<nav class="navbar navbar-expand-lg navbar-dark bg-primary">
    <div class="container-fluid">
        <a class="navbar-brand" href="panel.php">Cabañas Santiago - Clientes</a>
        <div class="d-flex">
            <span class="navbar-text me-3" style="font-size:0.9rem;">
                <?php echo htmlspecialchars($cli['nombre'] . ' ' . $cli['apellido']); ?>
                (DNI: <?php echo htmlspecialchars($cli['dni']); ?>)
                <?php if (!empty($cli['codigo_acceso'])): ?>
                    <br><small class="text-light">
                        Código de acceso: <?php echo htmlspecialchars($cli['codigo_acceso']); ?>
                    </small>
                <?php endif; ?>
            </span>
            <a class="btn btn-outline-light btn-sm" href="logout.php">Salir</a>
        </div>
    </div>
</nav>

<div class="container py-4">
    <h3 class="mb-3">Registrar mi compra</h3>
    <p class="text-muted" style="max-width:720px;">
        Completá estos datos para registrar la casa que compraste y cómo la estás pagando.
        Luego vas a poder ver el avance de los pagos y el saldo desde tu panel.
    </p>

    <?php if (!empty($errores)): ?>
        <div class="alert alert-danger">
            <ul class="mb-0">
                <?php foreach ($errores as $e): ?>
                    <li><?php echo htmlspecialchars($e); ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <form method="post" class="row g-3">
        <div class="col-md-8">
            <label class="form-label">Modelo / tipo de casa que compraste *</label>
            <input type="text" name="modelo" class="form-control"
                   value="<?php echo htmlspecialchars($datos['modelo']); ?>" required>
            <small class="text-muted">Ejemplo: Cabaña 2 dormitorios 54 m², Casa alpina, etc.</small>
        </div>

        <div class="col-md-4">
            <label class="form-label">Forma de venta *</label>
            <select name="forma_venta" class="form-select">
                <option value="contado" <?php echo $datos['forma_venta']==='contado'?'selected':''; ?>>Contado</option>
                <option value="financiado" <?php echo $datos['forma_venta']==='financiado'?'selected':''; ?>>Financiado</option>
                <option value="mixto" <?php echo $datos['forma_venta']==='mixto'?'selected':''; ?>>Mixto</option>
            </select>
        </div>

        <div class="col-md-3">
            <label class="form-label">Metros cubiertos (m²) *</label>
            <input type="number" step="0.01" min="0" name="metros_cubiertos" class="form-control"
                   value="<?php echo htmlspecialchars($datos['metros_cubiertos']); ?>" required>
        </div>
        <div class="col-md-3">
            <label class="form-label">Metros de galería (m²)</label>
            <input type="number" step="0.01" min="0" name="metros_galeria" class="form-control"
                   value="<?php echo htmlspecialchars($datos['metros_galeria']); ?>">
        </div>

        <div class="col-md-3">
            <label class="form-label">Sistema constructivo *</label>
            <select name="sistema" class="form-select">
                <option value="estructura" <?php echo $datos['sistema']==='estructura'?'selected':''; ?>>
                    Estructura
                </option>
                <option value="estructura_plataforma" <?php echo $datos['sistema']==='estructura_plataforma'?'selected':''; ?>>
                    Estructura + Plataforma
                </option>
                <option value="llave_en_mano" <?php echo $datos['sistema']==='llave_en_mano'?'selected':''; ?>>
                    Llave en mano
                </option>
            </select>
        </div>

        <div class="col-md-3">
            <label class="form-label">Precio total de la vivienda *</label>
            <input type="number" step="0.01" min="0" name="precio_lista" class="form-control"
                   value="<?php echo htmlspecialchars($datos['precio_lista']); ?>" required>
        </div>

        <div class="col-12">
            <hr>
            <h5>Entregas iniciales</h5>
        </div>

        <div class="col-md-3">
            <label class="form-label">Seña / entrega en efectivo</label>
            <input type="number" step="0.01" min="0" name="senia" class="form-control"
                   value="<?php echo htmlspecialchars($datos['senia']); ?>">
        </div>
        <div class="col-md-3">
            <label class="form-label">Entrega vehículo</label>
            <input type="number" step="0.01" min="0" name="entrega_vehiculo" class="form-control"
                   value="<?php echo htmlspecialchars($datos['entrega_vehiculo']); ?>">
        </div>
        <div class="col-md-3">
            <label class="form-label">Entrega terreno</label>
            <input type="number" step="0.01" min="0" name="entrega_terreno" class="form-control"
                   value="<?php echo htmlspecialchars($datos['entrega_terreno']); ?>">
        </div>
        <div class="col-md-3">
            <label class="form-label">Entrega cheques</label>
            <input type="number" step="0.01" min="0" name="entrega_cheques" class="form-control"
                   value="<?php echo htmlspecialchars($datos['entrega_cheques']); ?>">
        </div>

        <div class="col-12">
            <hr>
            <h5>Financiación (si corresponde)</h5>
        </div>

        <div class="col-md-3">
            <label class="form-label">Saldo a financiar</label>
            <input type="number" step="0.01" min="0" name="saldo_financiar" class="form-control"
                   value="<?php echo htmlspecialchars($datos['saldo_financiar']); ?>">
            <small class="text-muted">
                Si lo dejás vacío se calcula: precio total - entregas.
            </small>
        </div>
        <div class="col-md-3">
            <label class="form-label">Cantidad de cuotas</label>
            <input type="number" min="0" name="cuotas_cantidad" class="form-control"
                   value="<?php echo htmlspecialchars($datos['cuotas_cantidad']); ?>">
        </div>
        <div class="col-md-3">
            <label class="form-label">Monto de cada cuota</label>
            <input type="number" step="0.01" min="0" name="cuota_monto" class="form-control"
                   value="<?php echo htmlspecialchars($datos['cuota_monto']); ?>">
            <small class="text-muted">
                Si lo dejás vacío y hay saldo + cuotas, se calcula automáticamente.
            </small>
        </div>

        <div class="col-md-3">
            <label class="form-label">Tipo de plan</label>
            <select name="tipo_plan" class="form-select">
                <option value="ninguno" <?php echo $datos['tipo_plan']==='ninguno'?'selected':''; ?>>
                    Sin plan especial
                </option>
                <option value="plan_ahorro" <?php echo $datos['tipo_plan']==='plan_ahorro'?'selected':''; ?>>
                    Plan de ahorro
                </option>
            </select>
            <small class="text-muted">
                En planes de ahorro la obra inicia al llegar a un % de pagos.
            </small>
        </div>

        <div class="col-md-3">
            <label class="form-label">% para inicio de obra</label>
            <input type="number" step="0.01" min="0" max="100"
                   name="porcentaje_inicio_obra" class="form-control"
                   value="<?php echo htmlspecialchars($datos['porcentaje_inicio_obra']); ?>">
        </div>

        <div class="col-12 d-flex justify-content-between mt-3">
            <a href="panel.php" class="btn btn-secondary">Cancelar</a>
            <button type="submit" class="btn btn-primary">Guardar mi operación</button>
        </div>
    </form>
</div>
</body>
</html>
