<?php
// cliente/simulador.php
@session_start();
if (empty($_SESSION['cliente_id'])) {
    header('Location: login.php');
    exit;
}

require_once '../db.php';
$pdo = pdo();

$cliente_id = (int)$_SESSION['cliente_id'];

// Traer datos del cliente
$stmt = $pdo->prepare("SELECT * FROM clientes WHERE id = :id LIMIT 1");
$stmt->execute([':id' => $cliente_id]);
$cli = $stmt->fetch();
if (!$cli) {
    session_unset();
    session_destroy();
    header('Location: login.php');
    exit;
}

$errores = [];
$ok = false;

// Valores por defecto del form
$data = [
    'modelo'                  => '',
    'metros_cubiertos'        => '',
    'metros_galeria'          => '',
    'sistema'                 => 'estructura',
    'precio_referencia'       => '',
    'forma_venta_deseada'     => 'contado',
    'senia_deseada'           => '0',
    'entrega_vehiculo_deseada'=> '0',
    'entrega_vehiculo_detalle'=> '',
    'entrega_terreno_deseada' => '0',
    'entrega_terreno_detalle' => '',
    'entrega_cheques_deseada' => '0',
    'cuotas_deseadas'         => '0',
    'cuota_monto_estimado'    => '0',
    'comentarios_cliente'     => '',
];

// 🔹 Prefill desde modelos_casas si viene modelo_id por GET (y aún no se envió el form)
$modelo_id = isset($_GET['modelo_id']) ? (int)$_GET['modelo_id'] : 0;
if ($modelo_id > 0 && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    $stmtMod = $pdo->prepare("SELECT * FROM modelos_casas WHERE id = :id AND activo = 1 LIMIT 1");
    $stmtMod->execute([':id' => $modelo_id]);
    $mod = $stmtMod->fetch();
    if ($mod) {
        $data['modelo']           = $mod['nombre_modelo'];
        $data['metros_cubiertos'] = (float)$mod['metros_cubiertos'];
        $data['metros_galeria']   = (float)$mod['metros_galeria'];
        $data['sistema']          = $mod['sistema_default'];
        // Usamos precio_lista como referencia si está cargado, sino precio_desde
        $precio_ref = (float)$mod['precio_lista'] > 0 ? (float)$mod['precio_lista'] : (float)$mod['precio_desde'];
        $data['precio_referencia'] = $precio_ref > 0 ? $precio_ref : '';
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    foreach ($data as $k => $v) {
        if (isset($_POST[$k])) {
            $data[$k] = trim($_POST[$k]);
        }
    }

    // Validaciones básicas
    if ($data['modelo'] === '') {
        $errores[] = 'Indicá qué modelo de casa estás simulando.';
    }

    $precio_referencia = (float)($data['precio_referencia'] ?: 0);
    if ($precio_referencia <= 0) {
        $errores[] = 'El precio de referencia debe ser mayor a 0.';
    }

    $metros_cubiertos = (float)($data['metros_cubiertos'] ?: 0);
    if ($metros_cubiertos <= 0) {
        $errores[] = 'Indicá los metros cubiertos de la casa.';
    }

    $metros_galeria = (float)($data['metros_galeria'] ?: 0);

    $senia_deseada            = (float)($data['senia_deseada'] ?: 0);
    $entrega_vehiculo_deseada = (float)($data['entrega_vehiculo_deseada'] ?: 0);
    $entrega_terreno_deseada  = (float)($data['entrega_terreno_deseada'] ?: 0);
    $entrega_cheques_deseada  = (float)($data['entrega_cheques_deseada'] ?: 0);

    $cuotas_deseadas      = (int)($data['cuotas_deseadas'] ?: 0);
    $cuota_monto_estimado = (float)($data['cuota_monto_estimado'] ?: 0);

    if ($data['forma_venta_deseada'] !== 'contado' && $cuotas_deseadas > 0 && $cuota_monto_estimado <= 0) {
        $errores[] = 'Si indicás cantidad de cuotas, agregá un monto aproximado por cuota.';
    }

    if (empty($errores)) {
        try {
            $stmt = $pdo->prepare("
                INSERT INTO simulaciones_operacion
                (
                    cliente_id,
                    modelo, metros_cubiertos, metros_galeria, sistema,
                    precio_referencia,
                    forma_venta_deseada,
                    senia_deseada,
                    entrega_vehiculo_deseada, entrega_vehiculo_detalle,
                    entrega_terreno_deseada, entrega_terreno_detalle,
                    entrega_cheques_deseada,
                    cuotas_deseadas, cuota_monto_estimado,
                    comentarios_cliente,
                    estado,
                    created_at, updated_at
                )
                VALUES
                (
                    :cliente_id,
                    :modelo, :metros_cubiertos, :metros_galeria, :sistema,
                    :precio_referencia,
                    :forma_venta_deseada,
                    :senia_deseada,
                    :entrega_vehiculo_deseada, :entrega_vehiculo_detalle,
                    :entrega_terreno_deseada, :entrega_terreno_detalle,
                    :entrega_cheques_deseada,
                    :cuotas_deseadas, :cuota_monto_estimado,
                    :comentarios_cliente,
                    'pendiente',
                    NOW(), NOW()
                )
            ");

            $stmt->execute([
                ':cliente_id'               => $cliente_id,
                ':modelo'                   => $data['modelo'],
                ':metros_cubiertos'         => $metros_cubiertos,
                ':metros_galeria'           => $metros_galeria,
                ':sistema'                  => $data['sistema'],
                ':precio_referencia'        => $precio_referencia,
                ':forma_venta_deseada'      => $data['forma_venta_deseada'],
                ':senia_deseada'            => $senia_deseada,
                ':entrega_vehiculo_deseada' => $entrega_vehiculo_deseada,
                ':entrega_vehiculo_detalle' => $data['entrega_vehiculo_detalle'] ?: null,
                ':entrega_terreno_deseada'  => $entrega_terreno_deseada,
                ':entrega_terreno_detalle'  => $data['entrega_terreno_detalle'] ?: null,
                ':entrega_cheques_deseada'  => $entrega_cheques_deseada,
                ':cuotas_deseadas'          => $cuotas_deseadas,
                ':cuota_monto_estimado'     => $cuota_monto_estimado,
                ':comentarios_cliente'      => $data['comentarios_cliente'] ?: null,
            ]);

            $ok = true;

        } catch (PDOException $e) {
            $errores[] = 'Error al guardar la simulación: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Simulador de formas de pago - Cabañas Santiago</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">

<nav class="navbar navbar-expand-lg navbar-dark bg-primary">
    <div class="container-fluid">
        <a class="navbar-brand" href="panel.php">Cabañas Santiago - Clientes</a>
        <div class="d-flex">
            <span class="navbar-text me-3" style="font-size:0.9rem;">
                <?php echo htmlspecialchars($cli['nombre'] . ' ' . $cli['apellido']); ?>
                (DNI: <?php echo htmlspecialchars($cli['dni']); ?>)
                <?php if (!empty($cli['codigo_acceso'])): ?>
                    <br><small class="text-light">
                        Código de acceso: <?php echo htmlspecialchars($cli['codigo_acceso']); ?>
                    </small>
                <?php endif; ?>
            </span>
            <a class="btn btn-outline-light btn-sm" href="logout.php">Salir</a>
        </div>
    </div>
</nav>

<div class="container py-4">
    <h3 class="mb-3">Simulador de formas de pago</h3>

    <p class="text-muted" style="max-width: 760px;">
        Acá podés indicarnos <strong>cómo te gustaría pagar</strong> tu casa: contado, financiado o mixto.
        Esta información llega al equipo de Cabañas Santiago, que va a tomar contacto con vos para
        confirmar el plan final y generar el contrato.
    </p>

    <?php if ($ok): ?>
        <div class="alert alert-success">
            Tu simulación fue enviada. Un asesor de Cabañas Santiago va a revisar tu propuesta de pago y se
            va a comunicar con vos para confirmar los montos finales.
        </div>
    <?php endif; ?>

    <?php if (!empty($errores)): ?>
        <div class="alert alert-danger">
            <ul class="mb-0">
                <?php foreach ($errores as $e): ?>
                    <li><?php echo htmlspecialchars($e); ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <form method="post" class="row g-3">
        <div class="col-md-8">
            <label class="form-label">Modelo / tipo de casa *</label>
            <input type="text" name="modelo" class="form-control"
                   value="<?php echo htmlspecialchars($data['modelo']); ?>" required>
        </div>
        <div class="col-md-4">
            <label class="form-label">Sistema constructivo *</label>
            <select name="sistema" class="form-select">
                <option value="estructura" <?php echo $data['sistema']==='estructura'?'selected':''; ?>>
                    Estructura
                </option>
                <option value="estructura_plataforma" <?php echo $data['sistema']==='estructura_plataforma'?'selected':''; ?>>
                    Estructura + Plataforma
                </option>
                <option value="llave_en_mano" <?php echo $data['sistema']==='llave_en_mano'?'selected':''; ?>>
                    Llave en mano
                </option>
            </select>
        </div>

        <div class="col-md-3">
            <label class="form-label">Metros cubiertos (m²) *</label>
            <input type="number" step="0.01" min="0" name="metros_cubiertos" class="form-control"
                   value="<?php echo htmlspecialchars($data['metros_cubiertos']); ?>" required>
        </div>
        <div class="col-md-3">
            <label class="form-label">Metros galería (m²)</label>
            <input type="number" step="0.01" min="0" name="metros_galeria" class="form-control"
                   value="<?php echo htmlspecialchars($data['metros_galeria']); ?>">
        </div>
        <div class="col-md-6">
            <label class="form-label">Precio de referencia de la vivienda *</label>
            <input type="number" step="0.01" min="0" name="precio_referencia" class="form-control"
                   value="<?php echo htmlspecialchars($data['precio_referencia']); ?>" required>
        </div>

        <div class="col-12">
            <hr>
            <h5>Cómo te gustaría pagar</h5>
        </div>

        <div class="col-md-4">
            <label class="form-label">Forma de pago deseada *</label>
            <select name="forma_venta_deseada" class="form-select">
                <option value="contado" <?php echo $data['forma_venta_deseada']==='contado'?'selected':''; ?>>
                    Contado
                </option>
                <option value="financiado" <?php echo $data['forma_venta_deseada']==='financiado'?'selected':''; ?>>
                    Financiado
                </option>
                <option value="mixto" <?php echo $data['forma_venta_deseada']==='mixto'?'selected':''; ?>>
                    Mixto (contado + financiación)
                </option>
            </select>
        </div>

        <div class="col-md-4">
            <label class="form-label">Seña / entrega en efectivo</label>
            <input type="number" step="0.01" min="0" name="senia_deseada" class="form-control"
                   value="<?php echo htmlspecialchars($data['senia_deseada']); ?>">
        </div>

        <div class="col-12">
            <h6 class="mt-3">Entrega de vehículo (opcional)</h6>
        </div>
        <div class="col-md-3">
            <label class="form-label">Monto estimado del vehículo</label>
            <input type="number" step="0.01" min="0" name="entrega_vehiculo_deseada" class="form-control"
                   value="<?php echo htmlspecialchars($data['entrega_vehiculo_deseada']); ?>">
        </div>
        <div class="col-md-9">
            <label class="form-label">Detalle del vehículo</label>
            <input type="text" name="entrega_vehiculo_detalle" class="form-control"
                   placeholder="Marca, modelo, año, estado, etc."
                   value="<?php echo htmlspecialchars($data['entrega_vehiculo_detalle']); ?>">
        </div>

        <div class="col-12">
            <h6 class="mt-3">Entrega de terreno (opcional)</h6>
        </div>
        <div class="col-md-3">
            <label class="form-label">Monto estimado del terreno</label>
            <input type="number" step="0.01" min="0" name="entrega_terreno_deseada" class="form-control"
                   value="<?php echo htmlspecialchars($data['entrega_terreno_deseada']); ?>">
        </div>
        <div class="col-md-9">
            <label class="form-label">Detalle del terreno</label>
            <input type="text" name="entrega_terreno_detalle" class="form-control"
                   placeholder="Ubicación, medidas, etc."
                   value="<?php echo htmlspecialchars($data['entrega_terreno_detalle']); ?>">
        </div>

        <div class="col-12">
            <h6 class="mt-3">Cheques (opcional)</h6>
        </div>
        <div class="col-md-3">
            <label class="form-label">Monto total estimado en cheques</label>
            <input type="number" step="0.01" min="0" name="entrega_cheques_deseada" class="form-control"
                   value="<?php echo htmlspecialchars($data['entrega_cheques_deseada']); ?>">
        </div>

        <div class="col-12">
            <h6 class="mt-3">Financiación (si querés pagar en cuotas)</h6>
        </div>
        <div class="col-md-3">
            <label class="form-label">Cantidad de cuotas deseadas</label>
            <input type="number" min="0" name="cuotas_deseadas" class="form-control"
                   value="<?php echo htmlspecialchars($data['cuotas_deseadas']); ?>">
        </div>
        <div class="col-md-3">
            <label class="form-label">Monto aproximado por cuota</label>
            <input type="number" step="0.01" min="0" name="cuota_monto_estimado" class="form-control"
                   value="<?php echo htmlspecialchars($data['cuota_monto_estimado']); ?>">
        </div>

        <div class="col-12">
            <label class="form-label">Comentarios adicionales</label>
            <textarea name="comentarios_cliente" rows="3" class="form-control"
                      placeholder="Ej: fechas en las que podrías empezar, si tenés otra propuesta de pago, etc."><?php
                echo htmlspecialchars($data['comentarios_cliente']);
            ?></textarea>
        </div>

        <div class="col-12 d-flex justify-content-between mt-3">
            <a href="panel.php" class="btn btn-secondary">Volver al panel</a>
            <button type="submit" class="btn btn-primary">Enviar simulación</button>
        </div>
    </form>
</div>
</body>
</html>
