<?php
// admin_estado_puerto.php
@session_start();
require __DIR__ . '/db.php';

$pdo = null;
$msg = null;
$err = null;
$skipEdit = false;

try {
    if (function_exists('pdo')) {
        $pdo = pdo();

        // Tabla
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS estado_puerto (
              id INT AUTO_INCREMENT PRIMARY KEY,
              titulo VARCHAR(190) NOT NULL,
              estado ENUM('NORMAL','DEMORA','PARO','OTRO') NOT NULL DEFAULT 'NORMAL',
              descripcion TEXT,
              fecha DATE DEFAULT NULL,
              visible TINYINT(1) NOT NULL DEFAULT 1,
              created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
              updated_at TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");
    }
} catch (Throwable $e) {
    $err = 'Error al conectar o crear tabla: ' . $e->getMessage();
}

if (!function_exists('ep_s')) {
    function ep_s($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
}

/* ========= Acciones POST (crear / actualizar / toggle / borrar) ========= */
if ($pdo && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    try {
        if ($action === 'crear') {

            $titulo      = trim($_POST['titulo'] ?? '');
            $estado      = $_POST['estado'] ?? 'NORMAL';
            $descripcion = trim($_POST['descripcion'] ?? '');
            $fecha       = $_POST['fecha'] ?? null;
            $visible     = isset($_POST['visible']) ? 1 : 0;

            if ($titulo === '') {
                $err = 'El título es obligatorio.';
            } else {
                if ($fecha === '') $fecha = null;

                $stmt = $pdo->prepare("
                    INSERT INTO estado_puerto (titulo, estado, descripcion, fecha, visible)
                    VALUES (:titulo, :estado, :descripcion, :fecha, :visible)
                ");
                $stmt->execute([
                    ':titulo'      => $titulo,
                    ':estado'      => $estado,
                    ':descripcion' => $descripcion ?: null,
                    ':fecha'       => $fecha,
                    ':visible'     => $visible
                ]);
                $msg = 'Estado de puerto creado correctamente.';
            }

        } elseif ($action === 'update') {

            $id          = (int)($_POST['id'] ?? 0);
            $titulo      = trim($_POST['titulo'] ?? '');
            $estado      = $_POST['estado'] ?? 'NORMAL';
            $descripcion = trim($_POST['descripcion'] ?? '');
            $fecha       = $_POST['fecha'] ?? null;
            $visible     = isset($_POST['visible']) ? 1 : 0;

            if ($id <= 0) {
                $err = 'ID inválido para editar.';
            } elseif ($titulo === '') {
                $err = 'El título es obligatorio.';
            } else {
                if ($fecha === '') $fecha = null;

                $stmt = $pdo->prepare("
                    UPDATE estado_puerto
                    SET titulo = :titulo,
                        estado = :estado,
                        descripcion = :descripcion,
                        fecha = :fecha,
                        visible = :visible
                    WHERE id = :id
                ");
                $stmt->execute([
                    ':titulo'      => $titulo,
                    ':estado'      => $estado,
                    ':descripcion' => $descripcion ?: null,
                    ':fecha'       => $fecha,
                    ':visible'     => $visible,
                    ':id'          => $id
                ]);
                $msg = 'Estado de puerto actualizado correctamente.';
                $skipEdit = true; // después de guardar, volver a modo "nuevo"
            }

        } elseif ($action === 'toggle' && !empty($_POST['id'])) {

            $id  = (int)$_POST['id'];
            $vis = (int)($_POST['visible'] ?? 0);

            $stmt = $pdo->prepare("UPDATE estado_puerto SET visible = :v WHERE id = :id");
            $stmt->execute([':v' => $vis ? 1 : 0, ':id' => $id]);
            $msg = 'Visibilidad actualizada.';

        } elseif ($action === 'borrar' && !empty($_POST['id'])) {

            $id = (int)$_POST['id'];
            $stmt = $pdo->prepare("DELETE FROM estado_puerto WHERE id = :id");
            $stmt->execute([':id' => $id]);
            $msg = 'Registro eliminado.';
        }
    } catch (Throwable $e) {
        $err = 'Error al guardar: ' . $e->getMessage();
    }
}

/* ========= Registro a editar (GET ?edit=ID) ========= */
$editItem = null;
if ($pdo && !$skipEdit && isset($_GET['edit'])) {
    $editId = (int)$_GET['edit'];
    if ($editId > 0) {
        $stmt = $pdo->prepare("SELECT * FROM estado_puerto WHERE id = :id LIMIT 1");
        $stmt->execute([':id' => $editId]);
        $editItem = $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
    }
}

/* ========= Listado ========= */
$registros = [];
if ($pdo) {
    try {
        $q = $pdo->query("
            SELECT id, titulo, estado, descripcion, fecha, visible, created_at
            FROM estado_puerto
            ORDER BY fecha DESC, id DESC
        ");
        if ($q) {
            $registros = $q->fetchAll(PDO::FETCH_ASSOC);
        }
    } catch (Throwable $e) {
        $err = 'Error al listar registros: ' . $e->getMessage();
    }
}

/* ========= Valores para el formulario (nuevo / editar) ========= */
$formIsEdit  = $editItem !== null;
$formTitle   = $formIsEdit ? 'Editar estado' : 'Nuevo estado';
$formAction  = $formIsEdit ? 'update' : 'crear';
$f_id        = $editItem['id']         ?? '';
$f_titulo    = $editItem['titulo']     ?? '';
$f_estado    = $editItem['estado']     ?? 'NORMAL';
$f_fecha     = $editItem['fecha']      ?? '';
$f_desc      = $editItem['descripcion']?? '';
$f_visible   = isset($editItem['visible']) ? (int)$editItem['visible'] : 1;
?>

<style>
  /* Variables CSS para consistencia */
  :root {
    --primary: #4361ee;
    --primary-dark: #3a56d4;
    --secondary: #7209b7;
    --success: #2ecc71;
    --warning: #f39c12;
    --danger: #e74c3c;
    --dark: #2d3436;
    --light: #f8f9fa;
    --gray: #6c757d;
    --gray-light: #e9ecef;
    --border: rgba(0,0,0,0.08);
    --shadow: 0 10px 30px rgba(0,0,0,0.08);
    --shadow-lg: 0 15px 40px rgba(0,0,0,0.12);
    --radius: 12px;
    --radius-sm: 8px;
    --transition: all 0.3s ease;
  }

  /* Reset y estilos base */
  .ep-wrap * {
    box-sizing: border-box;
    margin: 0;
    padding: 0;
  }
  
  .ep-wrap {
    font-family: 'Segoe UI', system-ui, -apple-system, sans-serif;
    line-height: 1.6;
    color: var(--dark);
    background: #f5f7fa;
    min-height: 100vh;
    padding: 24px;
  }

  /* Encabezado */
  .ep-header {
    margin-bottom: 32px;
  }
  
  .ep-title {
    font-size: 32px;
    font-weight: 700;
    margin-bottom: 8px;
    background: linear-gradient(135deg, var(--primary), var(--secondary));
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    display: inline-block;
  }
  
  .ep-subtitle {
    font-size: 16px;
    color: var(--gray);
    max-width: 600px;
    line-height: 1.5;
  }

  /* Grid principal */
  .ep-grid {
    display: grid;
    grid-template-columns: minmax(0, 1fr) minmax(0, 1.5fr);
    gap: 28px;
    align-items: start;
  }
  
  @media (max-width: 1024px) {
    .ep-grid {
      grid-template-columns: 1fr;
    }
  }

  /* Tarjetas */
  .ep-card {
    background: #fff;
    border-radius: var(--radius);
    box-shadow: var(--shadow);
    overflow: hidden;
    transition: var(--transition);
    border: 1px solid var(--border);
  }
  
  .ep-card:hover {
    box-shadow: var(--shadow-lg);
    transform: translateY(-2px);
  }
  
  .ep-card-header {
    padding: 20px 24px 0;
    margin-bottom: 16px;
  }
  
  .ep-card-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--dark);
    display: flex;
    align-items: center;
    gap: 10px;
  }
  
  .ep-card-title::before {
    content: '';
    display: block;
    width: 4px;
    height: 20px;
    background: var(--primary);
    border-radius: 2px;
  }
  
  .ep-card-body {
    padding: 0 24px 24px;
  }

  /* Formularios */
  .ep-form-group {
    margin-bottom: 20px;
  }
  
  .ep-label {
    display: block;
    font-size: 14px;
    font-weight: 600;
    margin-bottom: 8px;
    color: var(--dark);
  }
  
  .ep-control, .ep-select, .ep-textarea {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid var(--border);
    border-radius: var(--radius-sm);
    font-size: 15px;
    font-family: inherit;
    transition: var(--transition);
    background: #fff;
  }
  
  .ep-control:focus, .ep-select:focus, .ep-textarea:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.15);
  }
  
  .ep-textarea {
    min-height: 100px;
    resize: vertical;
    line-height: 1.5;
  }
  
  .ep-form-row {
    display: flex;
    gap: 16px;
  }
  
  .ep-form-row > div {
    flex: 1;
  }
  
  @media (max-width: 640px) {
    .ep-form-row {
      flex-direction: column;
      gap: 0;
    }
  }

  /* Checkbox personalizado */
  .ep-checkbox {
    display: flex;
    align-items: center;
    gap: 10px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 500;
  }
  
  .ep-checkbox input {
    display: none;
  }
  
  .ep-checkbox .checkmark {
    width: 20px;
    height: 20px;
    border: 2px solid var(--border);
    border-radius: 5px;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: var(--transition);
  }
  
  .ep-checkbox input:checked + .checkmark {
    background: var(--primary);
    border-color: var(--primary);
  }
  
  .ep-checkbox input:checked + .checkmark::after {
    content: '✓';
    color: white;
    font-size: 12px;
    font-weight: bold;
  }

  /* Botones */
  .ep-btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
    padding: 10px 20px;
    border: none;
    border-radius: var(--radius-sm);
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition);
    text-decoration: none;
    font-family: inherit;
  }
  
  .ep-btn-primary {
    background: var(--primary);
    color: white;
    box-shadow: 0 4px 12px rgba(67, 97, 238, 0.3);
  }
  
  .ep-btn-primary:hover {
    background: var(--primary-dark);
    transform: translateY(-2px);
    box-shadow: 0 6px 16px rgba(67, 97, 238, 0.4);
  }
  
  .ep-btn-outline {
    background: transparent;
    color: var(--gray);
    border: 1px solid var(--border);
  }
  
  .ep-btn-outline:hover {
    background: var(--gray-light);
    color: var(--dark);
  }
  
  .ep-btn-danger {
    background: transparent;
    color: var(--danger);
    border: 1px solid var(--danger);
  }
  
  .ep-btn-danger:hover {
    background: var(--danger);
    color: white;
  }
  
  .ep-btn-sm {
    padding: 6px 12px;
    font-size: 12px;
  }
  
  .ep-actions {
    display: flex;
    gap: 8px;
    flex-wrap: wrap;
  }

  /* Badges de estado */
  .ep-badge {
    display: inline-block;
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 700;
    text-transform: uppercase;
    letter-spacing: 0.5px;
  }
  
  .ep-badge-normal { background: rgba(46, 204, 113, 0.15); color: #27ae60; }
  .ep-badge-demora { background: rgba(243, 156, 18, 0.15); color: #d35400; }
  .ep-badge-paro { background: rgba(231, 76, 60, 0.15); color: #c0392b; }
  .ep-badge-otro { background: rgba(108, 117, 125, 0.15); color: var(--gray); }

  /* Tabla */
  .ep-table-container {
    overflow: auto;
    border-radius: var(--radius-sm);
    border: 1px solid var(--border);
  }
  
  .ep-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
    min-width: 700px;
  }
  
  .ep-table thead {
    background: var(--light);
  }
  
  .ep-table th {
    padding: 14px 16px;
    text-align: left;
    font-weight: 600;
    color: var(--gray);
    font-size: 13px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    border-bottom: 1px solid var(--border);
  }
  
  .ep-table td {
    padding: 16px;
    border-bottom: 1px solid var(--border);
    vertical-align: top;
  }
  
  .ep-table tbody tr {
    transition: var(--transition);
  }
  
  .ep-table tbody tr:hover {
    background: rgba(67, 97, 238, 0.03);
  }
  
  .ep-table tbody tr:last-child td {
    border-bottom: none;
  }

  /* Alertas */
  .ep-alert {
    padding: 16px 20px;
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
    display: flex;
    align-items: center;
    gap: 12px;
    font-size: 14px;
    font-weight: 500;
  }
  
  .ep-alert-ok {
    background: rgba(46, 204, 113, 0.1);
    color: #27ae60;
    border-left: 4px solid #2ecc71;
  }
  
  .ep-alert-err {
    background: rgba(231, 76, 60, 0.1);
    color: #c0392b;
    border-left: 4px solid #e74c3c;
  }
  
  .ep-alert-icon {
    font-size: 18px;
  }

  /* Estado vacío */
  .ep-empty-state {
    text-align: center;
    padding: 40px 20px;
    color: var(--gray);
  }
  
  .ep-empty-state svg {
    width: 64px;
    height: 64px;
    margin-bottom: 16px;
    opacity: 0.5;
  }
  
  .ep-empty-state p {
    font-size: 16px;
    margin-bottom: 0;
  }

  /* Indicador de edición */
  .ep-editing-indicator {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    background: rgba(67, 97, 238, 0.1);
    color: var(--primary);
    padding: 8px 16px;
    border-radius: var(--radius-sm);
    font-size: 14px;
    font-weight: 500;
    margin-bottom: 20px;
  }
  
  .ep-editing-indicator::before {
    content: '';
    display: block;
    width: 10px;
    height: 10px;
    border-radius: 50%;
    background: var(--primary);
    animation: pulse 1.5s infinite;
  }
  
  @keyframes pulse {
    0% { opacity: 1; }
    50% { opacity: 0.5; }
    100% { opacity: 1; }
  }

  /* Responsive */
  @media (max-width: 768px) {
    .ep-wrap {
      padding: 16px;
    }
    
    .ep-title {
      font-size: 26px;
    }
    
    .ep-card-header, .ep-card-body {
      padding: 16px;
    }
    
    .ep-actions {
      flex-direction: column;
    }
    
    .ep-btn {
      width: 100%;
      justify-content: center;
    }
  }
</style>

<div class="ep-wrap">
  <header class="ep-header">
    <h1 class="ep-title">Estado de Puerto</h1>
    <p class="ep-subtitle">
      Administra y modera la información que se muestra en la sección 
      <strong>Estado de puerto</strong> del sitio público.
    </p>
  </header>

  <?php if ($msg): ?>
    <div class="ep-alert ep-alert-ok">
      <span class="ep-alert-icon">✓</span>
      <?php echo ep_s($msg); ?>
    </div>
  <?php endif; ?>

  <?php if ($err): ?>
    <div class="ep-alert ep-alert-err">
      <span class="ep-alert-icon">⚠</span>
      <?php echo ep_s($err); ?>
    </div>
  <?php endif; ?>

  <div class="ep-grid">
    <!-- Columna: formulario nuevo / editar estado -->
    <section class="ep-card">
      <div class="ep-card-header">
        <h2 class="ep-card-title"><?php echo ep_s($formTitle); ?></h2>
        <?php if ($formIsEdit): ?>
          <div class="ep-editing-indicator">
            Editando registro #<?php echo (int)$f_id; ?>
          </div>
        <?php endif; ?>
      </div>
      
      <div class="ep-card-body">
        <form method="post">
          <input type="hidden" name="action" value="<?php echo ep_s($formAction); ?>">
          <?php if ($formIsEdit): ?>
            <input type="hidden" name="id" value="<?php echo (int)$f_id; ?>">
          <?php endif; ?>

          <div class="ep-form-group">
            <label class="ep-label">Título *</label>
            <input type="text" name="titulo" class="ep-control" required
                   value="<?php echo ep_s($f_titulo); ?>" placeholder="Ingresa un título descriptivo">
          </div>

          <div class="ep-form-group">
            <div class="ep-form-row">
              <div>
                <label class="ep-label">Estado</label>
                <select name="estado" class="ep-select">
                  <option value="NORMAL" <?php echo $f_estado==='NORMAL'?'selected':''; ?>>Normal</option>
                  <option value="DEMORA" <?php echo $f_estado==='DEMORA'?'selected':''; ?>>Demora</option>
                  <option value="PARO"   <?php echo $f_estado==='PARO'  ?'selected':''; ?>>Paro</option>
                  <option value="OTRO"   <?php echo $f_estado==='OTRO'  ?'selected':''; ?>>Otro</option>
                </select>
              </div>
              <div>
                <label class="ep-label">Fecha (opcional)</label>
                <input type="date" name="fecha" class="ep-control"
                       value="<?php echo ep_s($f_fecha); ?>">
              </div>
            </div>
          </div>

          <div class="ep-form-group">
            <label class="ep-label">Descripción (opcional)</label>
            <textarea name="descripcion" class="ep-textarea" placeholder="Agrega detalles adicionales..."><?php echo ep_s($f_desc); ?></textarea>
          </div>

          <div class="ep-form-group" style="display: flex; justify-content: space-between; align-items: center;">
            <label class="ep-checkbox">
              <input type="checkbox" name="visible" <?php echo $f_visible ? 'checked' : ''; ?>>
              <span class="checkmark"></span>
              Publicar inmediatamente
            </label>

            <div style="display: flex; gap: 12px;">
              <?php if ($formIsEdit): ?>
                <a href="?" class="ep-btn ep-btn-outline">
                  Cancelar
                </a>
              <?php endif; ?>
              <button type="submit" class="ep-btn ep-btn-primary">
                <?php if ($formIsEdit): ?>
                  <span>✏️</span> Actualizar estado
                <?php else: ?>
                  <span>➕</span> Crear estado
                <?php endif; ?>
              </button>
            </div>
          </div>
        </form>
      </div>
    </section>

    <!-- Columna: listado de registros -->
    <section class="ep-card">
      <div class="ep-card-header">
        <h2 class="ep-card-title">Registros existentes</h2>
      </div>
      
      <div class="ep-card-body">
        <?php if (empty($registros)): ?>
          <div class="ep-empty-state">
            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
              <path d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
            </svg>
            <p>No hay registros cargados todavía.</p>
          </div>
        <?php else: ?>
          <div class="ep-table-container">
            <table class="ep-table">
              <thead>
                <tr>
                  <th>Fecha</th>
                  <th>Título</th>
                  <th>Estado</th>
                  <th>Visible</th>
                  <th>Acciones</th>
                </tr>
              </thead>
              <tbody>
              <?php foreach ($registros as $r):
                $estado = $r['estado'] ?? 'NORMAL';
                $badgeClass = 'ep-badge-normal';
                if ($estado === 'DEMORA') $badgeClass = 'ep-badge-demora';
                if ($estado === 'PARO')   $badgeClass = 'ep-badge-paro';
                if ($estado === 'OTRO')   $badgeClass = 'ep-badge-otro';
              ?>
                <tr>
                  <td>
                    <?php if (!empty($r['fecha'])): ?>
                      <strong><?php echo ep_s(date('d/m/Y', strtotime($r['fecha']))); ?></strong>
                    <?php else: ?>
                      <span style="color:var(--gray);">—</span>
                    <?php endif; ?>
                  </td>
                  <td>
                    <div style="font-weight: 600; margin-bottom: 4px;"><?php echo ep_s($r['titulo']); ?></div>
                    <?php if (!empty($r['descripcion'])): ?>
                      <div style="font-size: 13px; color: var(--gray); line-height: 1.4;">
                        <?php
                          echo nl2br(
                            ep_s(mb_strimwidth($r['descripcion'], 0, 100, '…', 'UTF-8'))
                          );
                        ?>
                      </div>
                    <?php endif; ?>
                  </td>
                  <td>
                    <span class="ep-badge <?php echo $badgeClass; ?>">
                      <?php echo ep_s($estado); ?>
                    </span>
                  </td>
                  <td>
                    <?php if (!empty($r['visible'])): ?>
                      <span style="color: var(--success); font-weight: 600;">Sí</span>
                    <?php else: ?>
                      <span style="color: var(--danger); font-weight: 600;">No</span>
                    <?php endif; ?>
                  </td>
                  <td>
                    <div class="ep-actions">
                      <!-- Editar -->
                      <a class="ep-btn ep-btn-outline ep-btn-sm" 
                         href="?<?php
                           $qs = $_GET;
                           $qs['edit'] = (int)$r['id'];
                           echo ep_s(http_build_query($qs));
                         ?>">
                        <span>✏️</span> Editar
                      </a>

                      <!-- Publicar / Ocultar -->
                      <form method="post">
                        <input type="hidden" name="action" value="toggle">
                        <input type="hidden" name="id" value="<?php echo (int)$r['id']; ?>">
                        <input type="hidden" name="visible" value="<?php echo empty($r['visible']) ? 1 : 0; ?>">
                        <button type="submit" class="ep-btn ep-btn-outline ep-btn-sm">
                          <?php if (empty($r['visible'])): ?>
                            <span>👁️</span> Publicar
                          <?php else: ?>
                            <span>👁️‍🗨️</span> Ocultar
                          <?php endif; ?>
                        </button>
                      </form>

                      <!-- Borrar -->
                      <form method="post" onsubmit="return confirm('¿Estás seguro de que querés eliminar este registro?');">
                        <input type="hidden" name="action" value="borrar">
                        <input type="hidden" name="id" value="<?php echo (int)$r['id']; ?>">
                        <button type="submit" class="ep-btn ep-btn-danger ep-btn-sm">
                          <span>🗑️</span> Borrar
                        </button>
                      </form>
                    </div>
                  </td>
                </tr>
              <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>
      </div>
    </section>
  </div>
</div>