<?php
// market_bootstrap.php
// Bootstrap general para el Marketplace Fetra

if (session_status() === PHP_SESSION_NONE) {
    @session_start();
}

require __DIR__ . '/db.php';

/**
 * Devuelve el PDO global del proyecto (usa tu función pdo() de db.php)
 */
function market_pdo(): PDO {
    static $pdo = null;
    if ($pdo === null) {
        $pdo = pdo();
    }
    return $pdo;
}

/**
 * Crea las tablas necesarias del marketplace si no existen
 */
function market_bootstrap_tables(): void {
    static $done = false;
    if ($done) return;
    $done = true;

    $pdo = market_pdo();

    // Tabla de usuarios del marketplace
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS market_usuarios (
            id INT AUTO_INCREMENT PRIMARY KEY,
            nombre VARCHAR(100) NOT NULL,
            apellido VARCHAR(100) NOT NULL,
            telefono VARCHAR(50) NOT NULL,
            email VARCHAR(150) NOT NULL UNIQUE,
            password_hash VARCHAR(255) NOT NULL,
            avatar VARCHAR(255) DEFAULT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME NULL,
            activo TINYINT(1) NOT NULL DEFAULT 1
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Tabla de productos
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS market_productos (
            id INT AUTO_INCREMENT PRIMARY KEY,
            usuario_id INT NOT NULL,
            titulo VARCHAR(200) NOT NULL,
            descripcion TEXT,
            ubicacion VARCHAR(200),
            precio DECIMAL(12,2) DEFAULT NULL,
            modo_precio ENUM('precio','consultar') NOT NULL DEFAULT 'precio',
            estado ENUM('activo','pausado','vendido') NOT NULL DEFAULT 'activo',
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME NULL,
            FOREIGN KEY (usuario_id) REFERENCES market_usuarios(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");

    // Tabla de fotos de producto
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS market_productos_fotos (
            id INT AUTO_INCREMENT PRIMARY KEY,
            producto_id INT NOT NULL,
            archivo VARCHAR(255) NOT NULL,
            es_portada TINYINT(1) NOT NULL DEFAULT 0,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (producto_id) REFERENCES market_productos(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
}

// Ejecutar bootstrap de tablas
market_bootstrap_tables();

/**
 * Helpers de sesión Marketplace
 */
function market_user_id(): ?int {
    return isset($_SESSION['market_user_id']) ? (int)$_SESSION['market_user_id'] : null;
}

function market_is_logged(): bool {
    return market_user_id() !== null;
}

function market_current_user(): ?array {
    static $user = null;
    if ($user !== null) return $user;

    $uid = market_user_id();
    if (!$uid) return null;

    $pdo = market_pdo();
    $st = $pdo->prepare("SELECT * FROM market_usuarios WHERE id = ? AND activo = 1 LIMIT 1");
    $st->execute([$uid]);
    $user = $st->fetch(PDO::FETCH_ASSOC) ?: null;
    return $user;
}

function market_require_login(): void {
    if (!market_is_logged()) {
        header("Location: login_market.php");
        exit;
    }
}

function market_redirect(string $url): void {
    header("Location: " . $url);
    exit;
}

/**
 * Sube un archivo (avatar o fotos) a /files/market/ y devuelve la ruta relativa o null
 * $inputName: nombre del input file
 * $multipleIndex: índice si es un input multiple (fotos[]), o null si es simple
 */
function market_upload_file(string $inputName, ?int $multipleIndex = null): ?string {
    // Si es multiple
    if ($multipleIndex !== null) {
        if (
            !isset($_FILES[$inputName]['name'][$multipleIndex]) ||
            $_FILES[$inputName]['error'][$multipleIndex] !== UPLOAD_ERR_OK
        ) {
            return null;
        }
        $name     = $_FILES[$inputName]['name'][$multipleIndex];
        $tmp_name = $_FILES[$inputName]['tmp_name'][$multipleIndex];
    } else {
        if (empty($_FILES[$inputName]['name']) || $_FILES[$inputName]['error'] !== UPLOAD_ERR_OK) {
            return null;
        }
        $name     = $_FILES[$inputName]['name'];
        $tmp_name = $_FILES[$inputName]['tmp_name'];
    }

    $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
    $allowed = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
    if (!in_array($ext, $allowed, true)) {
        return null;
    }

    $baseDir = __DIR__ . '/files/market/';
    if (!is_dir($baseDir)) {
        @mkdir($baseDir, 0775, true);
    }

    $filename = uniqid('mk_', true) . '.' . $ext;
    $dest = $baseDir . $filename;

    if (!@move_uploaded_file($tmp_name, $dest)) {
        return null;
    }

    // Devolvemos la ruta relativa para usar en src de <img>
    return 'files/market/' . $filename;
}
