<?php 
// perfil_market.php
require __DIR__ . '/market_bootstrap.php';

if (!market_is_logged()) {
    header('Location: login_market.php');
    exit;
}

$pdo = market_pdo();
$U   = market_current_user();
if (!$U) {
    header('Location: logout_market.php');
    exit;
}

$errores = [];
$ok      = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $nombre   = trim($_POST['nombre']   ?? '');
    $apellido = trim($_POST['apellido'] ?? '');
    $telefono = trim($_POST['telefono'] ?? '');
    $email    = trim($_POST['email']    ?? '');

    if ($nombre === '' || $apellido === '' || $telefono === '' || $email === '') {
        $errores[] = 'Completá nombre, apellido, teléfono y email.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errores[] = 'Email inválido.';
    } else {
        // Avatar opcional
        $avatar = $U['avatar'] ?? null;
        if (!empty($_FILES['avatar']['name'])) {
            $nuevoAvatar = market_upload_file('avatar', null);
            if ($nuevoAvatar) {
                $avatar = $nuevoAvatar;
            }
        }

        $st = $pdo->prepare("
            UPDATE market_usuarios
               SET nombre=?, apellido=?, telefono=?, email=?, avatar=?
             WHERE id=?
             LIMIT 1
        ");
        $st->execute([$nombre, $apellido, $telefono, $email, $avatar, $U['id']]);
        $ok = 'Datos actualizados correctamente.';
        $U  = market_current_user(true); // refrescar
    }
}

// Mis productos
$stP = $pdo->prepare("
    SELECT p.*,
           (SELECT archivo FROM market_productos_fotos f
             WHERE f.producto_id = p.id AND f.es_portada = 1
             ORDER BY f.id ASC LIMIT 1) AS portada
      FROM market_productos p
     WHERE p.usuario_id = ?
     ORDER BY p.created_at DESC
");
$stP->execute([$U['id']]);
$productos = $stP->fetchAll(PDO::FETCH_ASSOC);

function mk_s($v) {
    return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8');
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="utf-8">
    <title>Mi perfil - Marketplace</title>
    <?php require __DIR__ . '/_head_assets.php'; ?>
    <style>
        :root {
            --mk-primary: #0055a5;
            --mk-primary-light: #00a2ff;
            --mk-secondary: #6c757d;
            --mk-success: #198754;
            --mk-warning: #ffc107;
            --mk-danger: #dc3545;
            --mk-dark: #212529;
            --mk-light: #f8f9fc;
            --mk-border: #e0e4f0;
            --mk-shadow: 0 12px 30px rgba(0,0,0,0.08);
            --mk-shadow-hover: 0 18px 40px rgba(0,0,0,0.12);
            --mk-radius: 16px;
            --mk-radius-sm: 12px;
        }

        body {
            font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
        }

        .mk-main {
            padding: 2rem 0 3rem;
            background: linear-gradient(135deg, #f4f6fb 0%, #f0f4ff 100%);
            min-height: 100vh;
        }

        .mk-card {
            background: #fff;
            border-radius: var(--mk-radius);
            box-shadow: var(--mk-shadow);
            margin-bottom: 2rem;
            overflow: hidden;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            border: 1px solid rgba(255,255,255,0.2);
        }

        .mk-card:hover {
            transform: translateY(-3px);
            box-shadow: var(--mk-shadow-hover);
        }

        .mk-card-header {
            padding: 1.8rem 2rem;
            background: linear-gradient(135deg, var(--mk-primary), var(--mk-primary-light));
            color: #fff;
            position: relative;
            overflow: hidden;
        }

        .mk-card-header::before {
            content: '';
            position: absolute;
            top: -50%;
            right: -50%;
            width: 100%;
            height: 200%;
            background: linear-gradient(45deg, transparent, rgba(255,255,255,0.1), transparent);
            transform: rotate(45deg);
            animation: shine 3s infinite;
        }

        @keyframes shine {
            0% { transform: translateX(-100%) rotate(45deg); }
            100% { transform: translateX(100%) rotate(45deg); }
        }

        .mk-card-header h1 {
            font-size: 1.6rem;
            margin: 0;
            font-weight: 700;
            line-height: 1.3;
            position: relative;
            z-index: 2;
        }

        .mk-card-header p {
            margin: 0.4rem 0 0;
            font-size: 0.95rem;
            opacity: 0.95;
            font-weight: 500;
            position: relative;
            z-index: 2;
        }

        .mk-card-body {
            padding: 2rem;
        }

        /* ===== AVATAR MEJORADO ===== */
        .mk-avatar-section {
            display: flex;
            align-items: center;
            gap: 1.5rem;
            margin-bottom: 2rem;
            padding: 1.5rem;
            background: linear-gradient(135deg, #f8f9fc, #ffffff);
            border-radius: var(--mk-radius-sm);
            border: 1px solid var(--mk-border);
        }

        .mk-avatar-big {
            width: 100px;
            height: 100px;
            border-radius: 50%;
            background: linear-gradient(135deg, #e9ecef, #dee2e6);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 2.5rem;
            color: #777;
            overflow: hidden;
            border: 3px solid #fff;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            flex-shrink: 0;
        }

        .mk-avatar-big img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .mk-avatar-upload {
            flex: 1;
        }

        .mk-label {
            font-weight: 600;
            font-size: 0.9rem;
            color: var(--mk-dark);
            margin-bottom: 0.5rem;
            display: block;
        }

        /* ===== FORM CONTROLS ===== */
        .mk-form-control {
            border: 1.5px solid #e0e4f0;
            border-radius: var(--mk-radius-sm);
            padding: 0.75rem 1rem;
            font-size: 0.95rem;
            transition: all 0.3s ease;
            background: #fff;
            width: 100%;
        }

        .mk-form-control:focus {
            outline: none;
            border-color: var(--mk-primary);
            box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
            background: #fff;
        }

        .mk-form-control::placeholder {
            color: #9ca3af;
            font-size: 0.9rem;
        }

        .mk-help {
            font-size: 0.8rem;
            color: var(--mk-secondary);
            margin-top: 0.4rem;
            line-height: 1.4;
        }

        /* ===== BOTONES MEJORADOS ===== */
        .mk-actions {
            display: flex;
            flex-wrap: wrap;
            gap: 0.8rem;
            margin-top: 1.5rem;
        }

        .mk-btn {
            border: none;
            border-radius: var(--mk-radius-sm);
            font-weight: 600;
            padding: 0.8rem 1.8rem;
            font-size: 0.9rem;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            cursor: pointer;
            position: relative;
            overflow: hidden;
        }

        .mk-btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
            transition: left 0.5s;
        }

        .mk-btn:hover::before {
            left: 100%;
        }

        .mk-btn-primary {
            background: linear-gradient(135deg, var(--mk-primary), var(--mk-primary-light));
            color: #fff;
            box-shadow: 0 4px 15px rgba(0, 85, 165, 0.3);
        }

        .mk-btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0, 85, 165, 0.4);
            color: #fff;
        }

        .mk-btn-success {
            background: linear-gradient(135deg, var(--mk-success), #20c997);
            color: #fff;
            box-shadow: 0 4px 15px rgba(25, 135, 84, 0.3);
        }

        .mk-btn-success:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(25, 135, 84, 0.4);
            color: #fff;
        }

        .mk-btn-outline {
            background: transparent;
            border: 1.5px solid var(--mk-secondary);
            color: var(--mk-secondary);
        }

        .mk-btn-outline:hover {
            background: var(--mk-secondary);
            color: #fff;
            transform: translateY(-2px);
        }

        .mk-btn-icon {
            width: 16px;
            height: 16px;
        }

        /* ===== PRODUCTOS MEJORADOS ===== */
        .mk-products-grid {
            display: grid;
            gap: 1rem;
        }

        .mk-product-card {
            background: #fff;
            border-radius: var(--mk-radius-sm);
            box-shadow: 0 4px 12px rgba(0,0,0,0.05);
            border: 1px solid var(--mk-border);
            display: flex;
            align-items: stretch;
            overflow: hidden;
            transition: all 0.3s ease;
            height: 130px;
        }

        .mk-product-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        }

        .mk-product-media {
            flex: 0 0 140px;
            display: flex;
            align-items: center;
            justify-content: center;
            background: linear-gradient(135deg, #f8f9fc, #f0f4ff);
            border-right: 1px solid var(--mk-border);
            position: relative;
        }

        .mk-product-media img {
            max-width: 100%;
            max-height: 100px;
            width: auto;
            height: auto;
            object-fit: contain;
            transition: transform 0.3s ease;
        }

        .mk-product-card:hover .mk-product-media img {
            transform: scale(1.05);
        }

        .mk-product-img-empty {
            font-size: 0.8rem;
            color: var(--mk-secondary);
            text-align: center;
            padding: 0.5rem;
        }

        .mk-product-main {
            flex: 1 1 auto;
            display: flex;
            flex-direction: column;
            justify-content: space-between;
        }

        .mk-product-body {
            padding: 0.8rem 1rem 0.4rem;
            flex: 1;
        }

        .mk-product-title {
            font-weight: 700;
            font-size: 0.95rem;
            margin-bottom: 0.3rem;
            line-height: 1.2;
            color: var(--mk-dark);
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }

        .mk-product-meta {
            font-size: 0.8rem;
            color: var(--mk-secondary);
            line-height: 1.3;
        }

        .mk-product-price {
            font-weight: 600;
            color: var(--mk-success);
        }

        .mk-product-footer {
            padding: 0.4rem 1rem 0.8rem;
            border-top: 1px solid var(--mk-border);
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 0.75rem;
            gap: 0.5rem;
        }

        .mk-product-id {
            color: var(--mk-secondary);
            font-weight: 500;
        }

        .mk-product-actions {
            display: flex;
            gap: 0.3rem;
        }

        .mk-product-actions .btn {
            font-size: 0.7rem;
            padding: 0.2rem 0.5rem;
            border-radius: 6px;
            transition: all 0.3s ease;
        }

        .mk-product-actions .btn:hover {
            transform: translateY(-1px);
        }

        /* ===== BADGES MEJORADOS ===== */
        .mk-badge {
            display: inline-flex;
            align-items: center;
            gap: 0.3rem;
            padding: 0.3rem 0.6rem;
            border-radius: 50px;
            font-weight: 600;
            font-size: 0.7rem;
            text-transform: uppercase;
            letter-spacing: 0.3px;
        }

        .mk-badge-success {
            background: linear-gradient(135deg, #d1edff, #b6e0ff);
            color: #004085;
            border: 1px solid #66b3ff;
        }

        .mk-badge-warning {
            background: linear-gradient(135deg, #fff3cd, #ffeaa7);
            color: #856404;
            border: 1px solid #ffeaa7;
        }

        .mk-badge-secondary {
            background: linear-gradient(135deg, #e9ecef, #dee2e6);
            color: #495057;
            border: 1px solid #ced4da;
        }

        /* ===== ESTADO VACÍO ===== */
        .mk-empty-state {
            text-align: center;
            padding: 2rem;
            color: var(--mk-secondary);
        }

        .mk-empty-icon {
            font-size: 3rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }

        .mk-empty-title {
            font-size: 1.2rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
            color: var(--mk-dark);
        }

        .mk-empty-text {
            font-size: 0.9rem;
            margin-bottom: 1.5rem;
            line-height: 1.5;
        }

        /* ===== ALERTS MEJORADOS ===== */
        .mk-alert {
            border-radius: var(--mk-radius-sm);
            padding: 1rem 1.2rem;
            margin-bottom: 1.5rem;
            border: 1px solid transparent;
            font-size: 0.9rem;
        }

        .mk-alert-danger {
            background: linear-gradient(135deg, #f8d7da, #f5c6cb);
            border-color: #f1aeb5;
            color: #721c24;
        }

        .mk-alert-success {
            background: linear-gradient(135deg, #d1edff, #b6e0ff);
            border-color: #66b3ff;
            color: #004085;
        }

        /* ===== ANIMACIONES ===== */
        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .mk-card {
            animation: fadeInUp 0.6s ease-out;
        }

        /* ===== RESPONSIVE ===== */
        @media (max-width: 767.98px) {
            .mk-card-body {
                padding: 1.5rem;
            }
            
            .mk-card-header {
                padding: 1.5rem 1.5rem;
            }
            
            .mk-avatar-section {
                flex-direction: column;
                text-align: center;
                gap: 1rem;
            }
            
            .mk-actions {
                flex-direction: column;
            }
            
            .mk-actions .mk-btn {
                width: 100%;
                justify-content: center;
            }
            
            .mk-product-card {
                flex-direction: column;
                height: auto;
            }
            
            .mk-product-media {
                flex: 0 0 120px;
                border-right: none;
                border-bottom: 1px solid var(--mk-border);
            }
            
            .mk-product-footer {
                flex-direction: column;
                gap: 0.8rem;
                align-items: stretch;
            }
            
            .mk-product-actions {
                justify-content: center;
            }
        }

        @media (max-width: 575.98px) {
            .mk-card-body {
                padding: 1.2rem;
            }
            
            .mk-product-media {
                flex: 0 0 100px;
            }
        }
    </style>
</head>
<body>
<?php require __DIR__ . '/_navbar.php'; ?>

<main class="mk-main">
    <div class="container">
        <!-- PERFIL -->
        <div class="mk-card">
            <div class="mk-card-header">
                <h1>Mi perfil</h1>
                <p>Datos de contacto que verán otros camioneros en tus publicaciones.</p>
            </div>
            <div class="mk-card-body">
                <?php if ($errores): ?>
                    <div class="mk-alert mk-alert-danger">
                        <?php foreach ($errores as $e): ?>
                            <div><?= mk_s($e) ?></div>
                        <?php endforeach; ?>
                    </div>
                <?php elseif ($ok): ?>
                    <div class="mk-alert mk-alert-success"><?= mk_s($ok) ?></div>
                <?php endif; ?>

                <form method="post" enctype="multipart/form-data" id="profileForm">
                    <!-- Sección Avatar -->
                    <div class="mk-avatar-section">
                        <div class="mk-avatar-big" id="mkAvatarPreviewPerfil">
                            <?php if (!empty($U['avatar'])): ?>
                                <img src="<?= mk_s($U['avatar']) ?>" alt="Avatar">
                            <?php else: ?>
                                <span>👤</span>
                            <?php endif; ?>
                        </div>
                        <div class="mk-avatar-upload">
                            <label class="mk-label">Foto de perfil</label>
                            <input class="mk-form-control" type="file" name="avatar" id="avatarInputPerfil" accept="image/*">
                            <div class="mk-help">
                                Imagen opcional que se mostrará junto a tus publicaciones. Formatos: JPG, PNG, GIF
                            </div>
                        </div>
                    </div>

                    <!-- Campos del formulario -->
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="mk-label">Nombre</label>
                            <input class="mk-form-control" type="text" name="nombre"
                                   value="<?= mk_s($U['nombre']) ?>" 
                                   placeholder="Tu nombre"
                                   required>
                        </div>
                        <div class="col-md-6">
                            <label class="mk-label">Apellido</label>
                            <input class="mk-form-control" type="text" name="apellido"
                                   value="<?= mk_s($U['apellido']) ?>" 
                                   placeholder="Tu apellido"
                                   required>
                        </div>
                        <div class="col-md-6">
                            <label class="mk-label">Teléfono</label>
                            <input class="mk-form-control" type="text" name="telefono"
                                   value="<?= mk_s($U['telefono']) ?>" 
                                   placeholder="+54 9 11 1234-5678"
                                   required>
                        </div>
                        <div class="col-md-6">
                            <label class="mk-label">Email</label>
                            <input class="mk-form-control" type="email" name="email"
                                   value="<?= mk_s($U['email']) ?>" 
                                   placeholder="tu.email@ejemplo.com"
                                   required>
                        </div>
                    </div>

                    <!-- Acciones -->
                    <div class="mk-actions">
                        <button type="submit" class="mk-btn mk-btn-primary">
                            <span class="mk-btn-icon">💾</span>
                            Guardar cambios
                        </button>
                        <a href="nuevo_producto_market.php" class="mk-btn mk-btn-success">
                            <span class="mk-btn-icon">➕</span>
                            Publicar nuevo producto
                        </a>
                        <a href="market.php" class="mk-btn mk-btn-outline">
                            <span class="mk-btn-icon">←</span>
                            Volver al Marketplace
                        </a>
                    </div>
                </form>
            </div>
        </div>

        <!-- MIS PRODUCTOS -->
        <div class="mk-card">
            <div class="mk-card-header">
                <h1>Mis publicaciones</h1>
                <p>Productos y servicios que publicaste en el Marketplace.</p>
            </div>
            <div class="mk-card-body">
                <?php if (!$productos): ?>
                    <div class="mk-empty-state">
                        <div class="mk-empty-icon">📦</div>
                        <div class="mk-empty-title">Todavía no tenés publicaciones</div>
                        <div class="mk-empty-text">
                            Publicá tu primer producto o servicio para empezar a conectar con otros camioneros del marketplace.
                        </div>
                        <a href="nuevo_producto_market.php" class="mk-btn mk-btn-success">
                            <span class="mk-btn-icon">➕</span>
                            Publicar mi primer producto
                        </a>
                    </div>
                <?php else: ?>
                    <div class="mk-products-grid">
                        <?php foreach ($productos as $P): ?>
                            <?php
                            // Portada: intentamos usar la portada, si no existe buscamos otra foto válida
                            $img = $P['portada'] ?? '';
                            if ($img) {
                                $pathFs = __DIR__ . '/' . ltrim($img, '/');
                                if (!is_file($pathFs)) {
                                    $img = '';
                                }
                            }

                            if ($img === '' || $img === null) {
                                // Buscar cualquier otra foto que exista para este producto
                                $stFoto = $pdo->prepare("
                                    SELECT archivo 
                                    FROM market_productos_fotos 
                                    WHERE producto_id = ? 
                                    ORDER BY id ASC
                                ");
                                $stFoto->execute([(int)$P['id']]);
                                while ($F = $stFoto->fetch(PDO::FETCH_ASSOC)) {
                                    $cand = $F['archivo'] ?? '';
                                    if ($cand) {
                                        $pathFsCand = __DIR__ . '/' . ltrim($cand, '/');
                                        if (is_file($pathFsCand)) {
                                            $img = $cand;
                                            break;
                                        }
                                    }
                                }
                            }

                            // Estado de la publicación (badge)
                            $estado     = $P['estado'] ?? 'publicado';
                            $estadoTxt  = 'Publicado';
                            $estadoCss  = 'mk-badge-success';

                            if ($estado === 'pausado') {
                                $estadoTxt = 'Pausado';
                                $estadoCss = 'mk-badge-warning';
                            } elseif ($estado === 'borrado' || $estado === 'eliminado') {
                                $estadoTxt = 'Eliminado';
                                $estadoCss = 'mk-badge-secondary';
                            }

                            // Texto para botón de toggle
                            $toggleTxt = ($estado === 'pausado') ? 'Activar' : 'Pausar';
                            $toggleIcon = ($estado === 'pausado') ? '▶️' : '⏸️';
                            ?>
                            <div class="mk-product-card">
                                <div class="mk-product-media">
                                    <?php if ($img): ?>
                                        <img src="<?= mk_s($img) ?>" alt="<?= mk_s($P['titulo']) ?>">
                                    <?php else: ?>
                                        <div class="mk-product-img-empty">
                                            🖼️<br>Sin imagen
                                        </div>
                                    <?php endif; ?>
                                </div>

                                <div class="mk-product-main">
                                    <div class="mk-product-body">
                                        <div class="mk-product-title">
                                            <?= mk_s($P['titulo']) ?>
                                        </div>
                                        <div class="mk-product-meta">
                                            <?php if ($P['modo_precio'] === 'consultar'): ?>
                                                <span class="mk-product-price">Consultar precio</span>
                                            <?php elseif ($P['precio'] !== null && $P['precio'] !== ''): ?>
                                                <span class="mk-product-price">$ <?= number_format((float)$P['precio'], 0, ',', '.') ?></span>
                                            <?php endif; ?>
                                            <?php if (!empty($P['ubicacion'])): ?>
                                                · 📍 <?= mk_s($P['ubicacion']) ?>
                                            <?php endif; ?>
                                        </div>
                                        <div class="mk-product-meta">
                                            📅 Publicado el <?= date('d/m/Y', strtotime($P['created_at'] ?? 'now')) ?>
                                        </div>
                                    </div>

                                    <div class="mk-product-footer">
                                        <div class="d-flex align-items-center gap-2">
                                            <span class="mk-product-id">#<?= (int)$P['id'] ?></span>
                                            <span class="<?= $estadoCss ?>">
                                                <?= mk_s($estadoTxt) ?>
                                            </span>
                                        </div>
                                        <div class="mk-product-actions">
                                            <a href="ver_producto_market.php?id=<?= (int)$P['id'] ?>"
                                               class="btn btn-primary btn-sm" 
                                               title="Ver publicación">
                                                👁️Ver
                                            </a>
                                            <a href="editar_producto_market.php?id=<?= (int)$P['id'] ?>"
                                               class="btn btn-outline-secondary btn-sm" 
                                               title="Editar publicación">
                                                ✏️Editar
                                            </a>
                                            <!-- PAUSAR / ACTIVAR -->
    <a href="toggle_estado_producto_market.php?id=<?= (int)$P['id'] ?>"
       class="btn btn-outline-warning btn-sm" 
       title="<?= mk_s($toggleTxt) ?> publicación">
        <?= $toggleIcon ?> <?= $toggleTxt ?>
    </a>
                                            <a href="eliminar_producto_market.php?id=<?= (int)$P['id'] ?>"
                                               class="btn btn-outline-danger btn-sm"
                                               onclick="return confirm('¿Seguro que querés eliminar esta publicación?');"
                                               title="Eliminar publicación">
                                                🗑️Eliminar
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</main>

<?php require __DIR__ . '/_footer.php'; ?>
<?php require __DIR__ . '/_navbar_script.php'; ?>

<script>
document.addEventListener('DOMContentLoaded', function () {
    const input  = document.getElementById('avatarInputPerfil');
    const preview = document.getElementById('mkAvatarPreviewPerfil');
    if (!input || !preview) return;

    input.addEventListener('change', function () {
        const file = this.files && this.files[0];
        if (!file) return;
        
        // Validar tipo de archivo
        if (!file.type.startsWith('image/')) {
            alert('Por favor seleccioná una imagen válida (JPG, PNG, GIF).');
            this.value = '';
            return;
        }

        // Validar tamaño (max 5MB)
        if (file.size > 5 * 1024 * 1024) {
            alert('La imagen es demasiado grande. Máximo 5MB permitido.');
            this.value = '';
            return;
        }

        const url = URL.createObjectURL(file);
        preview.innerHTML = '';
        const img = document.createElement('img');
        img.src = url;
        preview.appendChild(img);
    });

    // Efectos hover en productos
    const productCards = document.querySelectorAll('.mk-product-card');
    productCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-2px)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });

    // Validación del formulario
    const profileForm = document.getElementById('profileForm');
    if (profileForm) {
        profileForm.addEventListener('submit', function(e) {
            const requiredFields = this.querySelectorAll('[required]');
            let valid = true;

            requiredFields.forEach(field => {
                if (!field.value.trim()) {
                    valid = false;
                    field.style.borderColor = '#dc3545';
                } else {
                    field.style.borderColor = '';
                }
            });

            if (!valid) {
                e.preventDefault();
                alert('Completá todos los campos obligatorios.');
            }
        });
    }
});
</script>
</body>
</html>