<?php
// ver_encuesta.php
require __DIR__ . '/db.php';
$pdo = pdo();

// ---------- Aseguramos tabla de respuestas ----------
$pdo->exec("
  CREATE TABLE IF NOT EXISTS encuestas_respuestas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    encuesta_id INT NOT NULL,
    opcion VARCHAR(255) NOT NULL,
    comentario TEXT NULL,
    nombre VARCHAR(100) NULL,
    apellido VARCHAR(100) NULL,
    email VARCHAR(150) NULL,
    ip VARCHAR(45) NULL,
    user_agent TEXT NULL,
    created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    INDEX (encuesta_id),
    INDEX (ip)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");

// ---------- Cargamos la encuesta ----------
$encuesta_id = (int)($_GET['id'] ?? $_POST['encuesta_id'] ?? 0);
if ($encuesta_id <= 0) {
    http_response_code(404);
    echo "Encuesta no encontrada.";
    exit;
}

$stmt = $pdo->prepare("
    SELECT id, titulo, descripcion, pregunta, opciones_json, permite_comentario, estado, created_at
    FROM encuestas
    WHERE id = ? LIMIT 1
");
$stmt->execute([$encuesta_id]);
$E = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$E || $E['estado'] !== 'publicada') {
    http_response_code(404);
    echo "Encuesta no disponible.";
    exit;
}

// Opciones
$opciones = [];
if (!empty($E['opciones_json'])) {
    $tmp = json_decode($E['opciones_json'], true);
    if (is_array($tmp)) {
        $opciones = array_values(array_filter(array_map('trim', $tmp), fn($v) => $v !== ''));
    }
}
if (!$opciones) {
    echo "La encuesta no tiene opciones configuradas.";
    exit;
}

// ---------- Chequeo por IP: ¿ya respondió esta encuesta? ----------
$ip = $_SERVER['REMOTE_ADDR'] ?? null;
$yaRespondio = false;

if ($ip) {
    $stmtCheck = $pdo->prepare("
        SELECT COUNT(*) 
        FROM encuestas_respuestas
        WHERE encuesta_id = ? AND ip = ?
    ");
    $stmtCheck->execute([$E['id'], $ip]);
    $yaRespondio = ($stmtCheck->fetchColumn() > 0);
}

$errores = [];
$ok = false;

// ---------- Procesar envío ----------
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Si ya había respondido desde esta IP, no dejamos volver a guardar
    if ($yaRespondio) {
        $errores[] = "Ya respondiste esta encuesta desde este dispositivo.";
    } else {
        $opcion   = trim($_POST['opcion'] ?? '');
        $coment   = trim($_POST['comentario'] ?? '');
        $nombre   = trim($_POST['nombre'] ?? '');
        $apellido = trim($_POST['apellido'] ?? '');
        $email    = trim($_POST['email'] ?? '');

        if ($opcion === '' || !in_array($opcion, $opciones, true)) {
            $errores[] = "Debe seleccionar una opción.";
        }

        if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errores[] = "El email ingresado no es válido.";
        }

        if (!$errores) {
            $ua  = $_SERVER['HTTP_USER_AGENT'] ?? '';

            $stmtIns = $pdo->prepare("
                INSERT INTO encuestas_respuestas
                (encuesta_id, opcion, comentario, nombre, apellido, email, ip, user_agent)
                VALUES (?,?,?,?,?,?,?,?)
            ");
            $stmtIns->execute([
                $E['id'],
                $opcion,
                $coment !== '' ? $coment : null,
                $nombre !== '' ? $nombre : null,
                $apellido !== '' ? $apellido : null,
                $email !== '' ? $email : null,
                $ip,
                $ua
            ]);

            $ok = true;
            $yaRespondio = true; // a partir de ahora también se considera respondida
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title><?= htmlspecialchars($E['titulo'], ENT_QUOTES, 'UTF-8') ?> · Fe.Tr.A</title>
  <?php require __DIR__ . '/_head_assets.php'; ?>
  <style>
    .encuesta-page{
      padding:40px 0 60px;
    }
    .encuesta-header h1{
      font-size:28px;
      margin:0 0 8px;
    }
    .encuesta-header p{
      margin:0 0 4px;
      color:var(--muted);
      max-width:720px;
    }
    .encuesta-meta{
      font-size:12px;
      color:var(--muted);
      margin-top:4px;
    }
    .encuesta-card{
      margin-top:24px;
      border-radius:16px;
      background:#fff;
      box-shadow:0 12px 30px rgba(15,26,36,.08);
      border:1px solid rgba(0,0,0,.05);
      padding:20px 20px 22px;
      max-width:720px;
    }
    .encuesta-question{
      font-weight:600;
      margin-bottom:14px;
    }
    .encuesta-options{
      margin-bottom:18px;
    }
    .encuesta-options label{
      display:flex;
      align-items:flex-start;
      gap:8px;
      font-size:14px;
      margin-bottom:6px;
      cursor:pointer;
    }
    .encuesta-options input[type="radio"]{
      margin-top:3px;
    }
    .encuesta-field{
      margin-bottom:12px;
    }
    .encuesta-field label{
      font-size:13px;
      font-weight:600;
      display:block;
      margin-bottom:4px;
    }
    .encuesta-field small{
      font-size:11px;
      color:var(--muted);
    }
    .encuesta-field input[type="text"],
    .encuesta-field input[type="email"],
    .encuesta-field textarea{
      width:100%;
      border-radius:10px;
      border:1px solid rgba(0,0,0,.16);
      padding:7px 9px;
      font-size:14px;
      font-family:inherit;
    }
    .encuesta-field textarea{
      min-height:80px;
      resize:vertical;
    }
    .btn-primary{
      border:none;
      border-radius:999px;
      background:var(--link);
      color:#fff;
      padding:8px 18px;
      font-size:14px;
      font-weight:600;
      cursor:pointer;
    }
    .alert{
      padding:8px 12px;
      border-radius:10px;
      font-size:13px;
      margin-bottom:12px;
    }
    .alert-ok{
      background:rgba(46,204,113,.08);
      color:#27ae60;
      border:1px solid rgba(46,204,113,.4);
    }
    .alert-info{
      background:rgba(52,152,219,.08);
      color:#2980b9;
      border:1px solid rgba(52,152,219,.4);
    }
    .alert-error{
      background:rgba(231,76,60,.08);
      color:#c0392b;
      border:1px solid rgba(231,76,60,.4);
    }
  </style>
</head>
<body>

  <?php require __DIR__ . '/_navbar.php'; ?>

  <div class="datebar">
    <div class="container inner">
      <span id="fecha-hoy">Hoy</span>
    </div>
  </div>

  <main>
    <section class="encuesta-page">
      <div class="container">

        <header class="encuesta-header">
          <h1><?= htmlspecialchars($E['titulo'], ENT_QUOTES, 'UTF-8') ?></h1>
          <?php if (!empty($E['descripcion'])): ?>
            <p><?= nl2br(htmlspecialchars($E['descripcion'], ENT_QUOTES, 'UTF-8')) ?></p>
          <?php endif; ?>
          <div class="encuesta-meta">
            Encuesta publicada el <?= date('d/m/Y', strtotime($E['created_at'])) ?>
          </div>
        </header>

        <div class="encuesta-card">
          <?php if ($ok): ?>
            <div class="alert alert-ok">
              ¡Gracias por participar! Tu respuesta fue registrada correctamente.
            </div>
          <?php elseif ($yaRespondio): ?>
            <div class="alert alert-info">
              Ya respondiste esta encuesta desde este dispositivo. Gracias por tu participación.
            </div>
          <?php endif; ?>

          <?php if ($errores): ?>
            <div class="alert alert-error">
              <?php foreach ($errores as $e): ?>
                <div><?= htmlspecialchars($e, ENT_QUOTES, 'UTF-8') ?></div>
              <?php endforeach; ?>
            </div>
          <?php endif; ?>

          <?php if (!$ok && !$yaRespondio): ?>
          <form method="post">
            <input type="hidden" name="encuesta_id" value="<?= (int)$E['id'] ?>">

            <div class="encuesta-question">
              <?= htmlspecialchars($E['pregunta'] ?: '¿Cuál es su opinión?', ENT_QUOTES, 'UTF-8') ?>
            </div>

            <div class="encuesta-options">
              <?php foreach ($opciones as $idx => $opt): ?>
                <label>
                  <input type="radio"
                         name="opcion"
                         value="<?= htmlspecialchars($opt, ENT_QUOTES, 'UTF-8') ?>"
                         required>
                  <span><?= htmlspecialchars($opt, ENT_QUOTES, 'UTF-8') ?></span>
                </label>
              <?php endforeach; ?>
            </div>

            <?php if (!empty($E['permite_comentario'])): ?>
              <div class="encuesta-field">
                <label for="comentario">Comentario adicional (opcional)</label>
                <textarea id="comentario" name="comentario"></textarea>
              </div>
            <?php endif; ?>

            <hr style="margin:14px 0 10px; border:none; border-top:1px dashed rgba(0,0,0,.1);">

            <div class="encuesta-field">
              <label>Datos de contacto (opcionales)</label>
              <small>Podés responder de forma anónima. Si completás tus datos podremos contactarte ante alguna gestión puntual.</small>
            </div>

            <div class="encuesta-field">
              <label for="nombre">Nombre (opcional)</label>
              <input type="text" id="nombre" name="nombre">
            </div>

            <div class="encuesta-field">
              <label for="apellido">Apellido (opcional)</label>
              <input type="text" id="apellido" name="apellido">
            </div>

            <div class="encuesta-field">
              <label for="email">Email (opcional)</label>
              <input type="email" id="email" name="email" placeholder="tucorreo@ejemplo.com">
            </div>

            <button type="submit" class="btn-primary">Enviar respuesta</button>
          </form>
          <?php endif; ?>
        </div>

      </div>
    </section>
  </main>

  <?php require __DIR__ . '/_footer.php'; ?>
  <?php require __DIR__ . '/_navbar_script.php'; ?>
</body>
</html>
