<?php
/**
 * mapa_mesas_vertical.php  – edición limpia 20‑Jul‑2025
 * --------------------------------------------------------
 * ▸ Mapa ARRIBA (arrastrable)
 * ▸ Inventario + Config. ABAJO
 * ▸ Movés las mesas libremente → «Guardar posiciones» persiste todo en BD.
 * ▸ Al refrescar, se cargan desde BD y quedan fijas.
 */

session_start();
require 'conexion.php';

date_default_timezone_set('America/Argentina/Buenos_Aires');

// ⚙️ Ajustes rápidos
const IFRAME_H = 420; // alto total del iframe
const MAP_H    = 300; // alto visible del mapa
const MESA_SZ  = 32;  // tamaño del bloque mesa
const INV_W    = 60;  // ancho inventario

/* ---------- Garantizar columnas (una sola vez) ---------- */
function ensure($con,$col,$sql){
  if(!mysqli_num_rows(mysqli_query($con,"SHOW COLUMNS FROM mesas LIKE '$col'")))
      mysqli_query($con,$sql);
}
ensure($con,'pos_x', "ALTER TABLE mesas ADD COLUMN pos_x INT UNSIGNED NOT NULL DEFAULT 0");
ensure($con,'pos_y', "ALTER TABLE mesas ADD COLUMN pos_y INT UNSIGNED NOT NULL DEFAULT 0");
ensure($con,'placed',"ALTER TABLE mesas ADD COLUMN placed TINYINT(1) NOT NULL DEFAULT 0");

/* ---------- Guardado individual (drag‑and‑drop live) ---------- */
if(isset($_POST['mesa_id'])){
  header('Content-Type: application/json');
  $id = intval($_POST['mesa_id']);
  $x  = intval($_POST['x'] ?? 0);
  $y  = intval($_POST['y'] ?? 0);
  $p  = intval($_POST['placed']);
  $st = $con->prepare("UPDATE mesas SET pos_x=?, pos_y=?, placed=? WHERE id=? LIMIT 1");
  $st->bind_param('iiii', $x, $y, $p, $id);
  echo json_encode(['success' => $st->execute()]);
  exit;
}

/* ---------- Guardado MASIVO desde botón ---------- */
if(isset($_POST['save_bulk'])){
  header('Content-Type: application/json');
  $arr = json_decode($_POST['data'] ?? '[]', true);
  if(!$arr || !is_array($arr)){
    echo json_encode(['success'=>false,'msg'=>'Datos vacíos']);
    exit;
  }

  // Iniciar transacción (si InnoDB)
  $supportsTx = ($con->query("SHOW VARIABLES LIKE 'tx_isolation'")->num_rows > 0);
  if($supportsTx) $con->begin_transaction();

  $st = $con->prepare("UPDATE mesas SET pos_x=?, pos_y=?, placed=? WHERE id=? LIMIT 1");
  foreach($arr as $m){
    $x = intval($m['x']);
    $y = intval($m['y']);
    $p = intval($m['placed']);
    $id= intval($m['id']);
    $st->bind_param('iiii',$x,$y,$p,$id);
    if(!$st->execute()){
      if($supportsTx) $con->rollback();
      echo json_encode(['success'=>false,'msg'=>'DB error: '.mysqli_error($con)]);
      exit;
    }
  }
  if($supportsTx) $con->commit();
  echo json_encode(['success'=>true,'count'=>count($arr)]);
  exit;
}

/* ---------- Datos para dibujar ---------- */
$mesas = mysqli_fetch_all(mysqli_query($con,
  "SELECT id,nombre,pos_x,pos_y,placed FROM mesas ORDER BY id"), MYSQLI_ASSOC);
$inv   = array_filter($mesas, fn($m)=>$m['placed']==0);
$plano = array_filter($mesas, fn($m)=>$m['placed']==1);
?>
<!DOCTYPE html>
<html lang="es">
<head>
<meta charset="utf-8">
<title>Mapa Mesas</title>
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
<link rel="stylesheet" href="https://code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css">
<style>
  :root{ --mesa:<?=MESA_SZ?>px; --inv:<?=INV_W?>px; }
  html,body{margin:0;height:100%;overflow:hidden;background:#eef5ff;font-size:12px;}
  #wrap{display:flex;flex-direction:column;height:<?=IFRAME_H?>px;}
  /* mapa */
  #mapa{height:<?=MAP_H?>px;background:url('img/floorplan.svg') center/contain no-repeat;position:relative;overflow:hidden;}
  /* zona inferior */
  #zona{flex:1;display:flex;border-top:1px solid #ccd;}
  #inv{width:var(--inv);padding:8px;border-right:1px solid #ccd;overflow:auto;}
  #cfg{flex:1;padding:8px;overflow:auto;}
  /* mesa */
  .mesa{width:var(--mesa);height:var(--mesa);background:#90caf9;border-radius:5px;display:flex;align-items:center;justify-content:center;font-weight:600;color:#08326b;cursor:grab;user-select:none;box-shadow:0 1px 3px rgba(0,0,0,.3);transition:transform .12s;}
  .mesa:hover{transform:scale(1.06);}
  #inv .mesa{margin:4px auto;position:relative;}
  .lbl{position:absolute;top:2px;left:4px;font-size:10px;color:#555;}
</style>
</head>
<body>
<div id="wrap">
  <!-- 🗺️ MAPA -->
  <div id="mapa">
    <?php foreach($plano as $m): ?>
      <div class="mesa" data-id="<?=$m['id']?>" style="left:<?=$m['pos_x']?>px; top:<?=$m['pos_y']?>px; position:absolute;">
        <?=htmlspecialchars($m['nombre'])?>
      </div>
    <?php endforeach; ?>
  </div>

  <!-- 🔻 Inventario + Config -->
  <div id="zona">
    <div id="inv"><span class="lbl">Inv.</span>
      <?php foreach($inv as $m): ?>
        <div class="mesa" data-id="<?=$m['id']?>"><?=htmlspecialchars($m['nombre'])?></div>
      <?php endforeach; ?>
    </div>
    <div id="cfg">
      <button id="btnGuardar" class="btn btn-sm btn-primary mb-2">💾 Guardar posiciones</button>
      <div class="small text-muted">Arrastrá las mesas y luego pulsa el botón para persistir.</div>
    </div>
  </div>
</div>

<!-- modal realtime -->
<div class="modal fade" id="modalMesa" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-centered">
    <div class="modal-content bg-dark text-white">
      <div class="modal-header border-0">
        <h5 class="modal-title">Mesa <span id="modalNum"></span></h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body p-0">
        <iframe id="modalIframe" src="" style="width:100%;height:70vh;border:0;"></iframe>
      </div>
    </div>
  </div>
</div>

<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://code.jquery.com/ui/1.13.2/jquery-ui.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
$(function(){
  // helper para draggable
  function makeDrag($e){
    $e.draggable({helper:'clone',revert:'invalid',start:()=>{$e.css('opacity',.4)},stop:()=>{$e.css('opacity',1)}});
  }
  makeDrag($('.mesa'));

  // drop en mapa
  $('#mapa').droppable({accept:'.mesa',drop:function(e,ui){const $m=ui.draggable;const mapOffset=$(this).offset();$m.appendTo('#mapa');$m.css({position:'absolute',left:ui.offset.left-mapOffset.left,top:ui.offset.top-mapOffset.top});makeDrag($m);}});
  // drop en inventario
  $('#inv').droppable({accept:'.mesa',drop:function(e,ui){const $m=ui.draggable;$m.css({position:'relative',left:'',top:''}).appendTo('#inv');makeDrag($m);}});

  // abrir modal realtime al click
  $(document).on('click','.mesa',function(){const id=$(this).data('id');$('#modalNum').text(id);$('#modalIframe').attr('src',`mesa_${id}.php?embed=1&ts=${Date.now()}`);bootstrap.Modal.getOrCreateInstance('#modalMesa').show();});

  // Guardar posiciones en bloque
  $('#btnGuardar').on('click',function(){const arr=[];$('.mesa').each(function(){const $m=$(this);const id=$m.data('id');const placed=$m.parent().attr('id')==='mapa'?1:0;let x=0,y=0;if(placed){x=parseInt($m.css('left'))||0;y=parseInt($m.css('top'))||0;}arr.push({id,x,y,placed});});
    $.post(location.pathname,{save_bulk:1,data:JSON.stringify(arr)},function(d){if(d.success){alert('Posiciones guardadas ✅');}else{alert('Error: '+(d.msg||'desconocido'));}},'json');});
});
// Ya debés tener algo así al crear el modal
const miModal = new bootstrap.Modal(document.getElementById('modalMesa'), {});

// Cuando el usuario lo cierra:
document.getElementById('modalMesa').addEventListener('hidden.bs.modal', () => {
  window.top.postMessage({ action: 'modal_closed' }, '*');
});

</script>
</body>
</html>
